/**
 * @license
 * Copyright 2017 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { Path } from './util/Path';
import { Node } from './snap/Node';
/**
 * Helper class to store a sparse set of snapshots.
 */
export declare class SparseSnapshotTree {
    private value;
    private readonly children;
    /**
     * Gets the node stored at the given path if one exists.
     *
     * @param path Path to look up snapshot for.
     * @return The retrieved node, or null.
     */
    find(path: Path): Node | null;
    /**
     * Stores the given node at the specified path. If there is already a node
     * at a shallower path, it merges the new data into that snapshot node.
     *
     * @param path Path to look up snapshot for.
     * @param data The new data, or null.
     */
    remember(path: Path, data: Node): void;
    /**
     * Purge the data at path from the cache.
     *
     * @param path Path to look up snapshot for.
     * @return True if this node should now be removed.
     */
    forget(path: Path): boolean;
    /**
     * Recursively iterates through all of the stored tree and calls the
     * callback on each one.
     *
     * @param prefixPath Path to look up node for.
     * @param func The function to invoke for each tree.
     */
    forEachTree(prefixPath: Path, func: (a: Path, b: Node) => unknown): void;
    /**
     * Iterates through each immediate child and triggers the callback.
     *
     * @param func The function to invoke for each child.
     */
    forEachChild(func: (a: string, b: SparseSnapshotTree) => void): void;
}
