//
//  BingoDatabase.m
//  BingoGame
//
//  Created by Jenny Wu on 11/02/14.
//  Copyright (c) 2014 Jenny Wu. All rights reserved.
//

#import "BingoDatabase.h"
#import "Utility.h"

@interface BingoDatabase()
//private method declaration
-(NSDictionary *) gamesTableColumnsDef;
-(NSDictionary *) imageProofsTableColumnsDef;
-(int)createTableWithName:(NSString *) name andColDef:(NSDictionary*) colDef;

@end


@implementation BingoDatabase
//readonly property with custom getter are not synthesize automatically. i.e. _databaseFilePath will be undefined if we don' explicitly @synthesize the property here.
@synthesize databaseFilePath = _databaseFilePath;
@synthesize database = _database;

//Game table and columns
NSString *const gamesTbName = @"Games";
NSString *const COL_STARTTIME = @"STARTTIME";
NSString *const COL_ENDTIME = @"ENDTIME";
NSString *const COL_LAYOUT = @"LAYOUT";
NSString *const COL_STATE = @"STATE";
NSString *const COL_WON = @"WON";;

//Image Proof table and columns
NSString *const imageProofsTbName = @"ImageProofs";
NSString *const COL_GAME_ID = @"GAME_ID";
NSString *const COL_DATE_TAKEN = @"DATE_TAKEN";
NSString *const COL_SIGNID = @"SIGNID";
NSString *const COL_LONGITUDE = @"LONGITUDE";
NSString *const COL_LATITUDE = @"LATITUDE";
NSString *const COL_FILEPATH = @"FILEPATH";
NSString *const COL_VERIFIED = @"VERIFIED";

NSString *const COMMA_SEPARATOR = @",";


//returns the databasefile's full path includeing the file name with extension
-(NSString *)databaseFilePath
{
    //this also works if _databaseFilePath is nil, calling lenght on nil will return 0
    if([_databaseFilePath length] == 0){
        NSArray *dirPaths = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
        
        //documents directory
        NSString *dbPath = [dirPaths objectAtIndex:0];
        
        _databaseFilePath = [dbPath stringByAppendingPathComponent:@"RoadSignBingo.db"];
    }
    return _databaseFilePath;
}

//Singleton to open the RoadSignBingo.sql database, if the database does not exist.
//it will be created and then opened
-(sqlite3 *)database{
    //if we have not got a handle to the database yet, get it
    if(!_database){
        //Opens specified database file. If the database file does not already exist, it is created.
        const char *dbfile = [[self databaseFilePath] UTF8String];
        if (sqlite3_open(dbfile, &_database) != SQLITE_OK){
            NSLog(@"Failed to open RoadSignBingo.sql at %@. Error: %s\n", [self databaseFilePath], sqlite3_errmsg(_database));
            sqlite3_close(_database); //even if there is an error, _database is still assigned.
            _database = NULL;
        }
    }
    return _database;
}

-(BOOL)createDatabaseSchema
{
    int result = NO;
    if ([self database]) {
        result = [self createTableWithName:gamesTbName andColDef:[self gamesTableColumnsDef]];
        if (result == SQLITE_OK){
            result = [self createTableWithName:imageProofsTbName andColDef:[self imageProofsTableColumnsDef]];
        }
        sqlite3_close(_database);
        _database = NULL;
    }
    return (result == SQLITE_OK) ? YES : NO;
}

-(int)createTableWithName:(NSString *)name andColDef:(NSArray *)colDef
{
    int result = -1;
    if ([colDef count] > 0){
        NSMutableString * colDefSQL = [[NSMutableString alloc] init];

        for (NSArray *subArray in colDef) {
            [colDefSQL appendFormat:@"%@ %@,", subArray[0], subArray[1]];
        }
//        for (NSString *key in colDef) {
//            [colDefSQL appendFormat:@"%@ %@,", key, [colDef valueForKey:key]];
//        }
        //remove last comma character
        [colDefSQL deleteCharactersInRange:NSMakeRange(colDefSQL.length-1, 1)];
        
        NSString *createTbSQL = [NSString stringWithFormat:
                                 @"CREATE TABLE IF NOT EXISTS %@"
                                 "(%@)", name, colDefSQL];
        //NSLog(@"%@", createTbSQL);
        
        char *errMsg = NULL;
        result = sqlite3_exec(_database, [createTbSQL UTF8String], NULL, NULL, &errMsg);
        if (result!= SQLITE_OK) {
            NSLog(@"FAIL TO CREATE %@ table. ERROR Code %i: %s", name, result,errMsg);
        }
        
    }
    return result;
}


-(NSArray *)gamesTableColumnsDef
{
    NSArray *colDef = @[
                         @[@"ID", @"INTEGER PRIMARY KEY AUTOINCREMENT"],
                         @[@"STARTTIME", @"DATETIME"],
                         @[@"ENDTIME", @"DATETIME"],
                         @[@"LAYOUT", @"TEXT"],
                         @[@"STATE", @"INTEGER"],
                         @[@"WON", @"INTEGER"]
                         ];
    return colDef;
}


-(NSArray *)imageProofsTableColumnsDef
{
    NSArray *colDef = @[
                        @[@"ID", @"INTEGER PRIMARY KEY AUTOINCREMENT"],
                        @[@"GAME_ID", @"INTEGER"],
                        @[@"DATE_TAKEN", @"DATETIME"],
                        @[@"SIGNID", @"INTEGER"],
                        @[@"LONGITUDE", @"REAL"],
                        @[@"LATITUDE", @"REAL"],
                        @[@"FILEPATH", @"TEXT"],
                        @[@"VERIFIED",@"INTEGER"],
                        ];
    return colDef;
}

-(BOOL)executeSQLStatement:(const char *)sql_stmt
{
    sqlite3_stmt *complied_stmt;
    BOOL result = NO;
    int resultNO;
    
    if(self.database){
        resultNO = sqlite3_prepare_v2(self.database, sql_stmt, -1, &complied_stmt, NULL);
        if(resultNO == SQLITE_OK){
            resultNO = sqlite3_step(complied_stmt);
            if(resultNO == SQLITE_DONE){
                result = YES;
            }
        }else{
            [Utility showAlertMessageWithTitle:@"why" message:[NSString stringWithUTF8String:sqlite3_errmsg(self.database)] delegate:nil canceltButtonText:@"OK"];
        }
        sqlite3_finalize(complied_stmt);
        sqlite3_close(self.database);
        _database = NULL;
    }
    return result;
}

-(NSNumber *)maxRowIDForTable:(NSString *) tableName
{
    NSNumber *_ID;
    if(self.database){
        NSString *querySQL = [NSString stringWithFormat:@"SELECT MAX(ID) FROM %@", tableName];
        const char *query_stmt = [querySQL UTF8String];
        sqlite3_stmt *compiled_sql;
        if(sqlite3_prepare_v2(self.database, query_stmt, -1, &compiled_sql, NULL) == SQLITE_OK){
            if(sqlite3_step(compiled_sql) == SQLITE_ROW){
                _ID = [NSNumber numberWithUnsignedLong: sqlite3_column_int64(compiled_sql,0)];
            }
        }
        sqlite3_finalize(compiled_sql);
        sqlite3_close(self.database);
        _database = NULL;
    }
    return _ID;
}


//Gameboard related operations
-(NSNumber *)maxRowIDForGameBoard{
    return [self maxRowIDForTable:gamesTbName];
}

-(Gameboard *)getLastGame
{
    Gameboard *game;
    if(self.database){
        NSString *querySQL = [NSString stringWithFormat:@"SELECT * FROM %@ ORDER BY ID DESC LIMIT 1", gamesTbName];
        const char *query_stmt = [querySQL UTF8String];
        sqlite3_stmt *compiled_sql;
        if(sqlite3_prepare_v2(self.database, query_stmt, -1, &compiled_sql, NULL) == SQLITE_OK){
            if(sqlite3_step(compiled_sql) == SQLITE_ROW){
                NSNumber *ID = [NSNumber numberWithUnsignedLong: sqlite3_column_int64(compiled_sql,0)];
                
                NSString *startTimeStr = [[NSString alloc] initWithUTF8String:(const char *)sqlite3_column_text(compiled_sql, 1)];
                NSDate *startTime = [Utility dateFromDatabaseString:startTimeStr];
                
                NSString *endTimeStr = [[NSString alloc] initWithUTF8String:(const char *)sqlite3_column_text(compiled_sql, 2)];
                NSDate *endTime = [Utility dateFromDatabaseString:endTimeStr];
                
                NSString *layoutStr = [[NSString alloc] initWithUTF8String:(const char *)sqlite3_column_text(compiled_sql, 3)];
                NSArray *layout = [layoutStr componentsSeparatedByString:COMMA_SEPARATOR];
                
                NSNumber *status = [NSNumber numberWithUnsignedLong: sqlite3_column_int64(compiled_sql,4)];
                
                game = [[Gameboard alloc] initExistingGameWithLayout:layout status:status StartTime:startTime EndTime:endTime ID:ID];

            }
        }
        sqlite3_finalize(compiled_sql);
        sqlite3_close(self.database);
        _database = NULL;
    }
    game.dictionaryImageProofs = [self getImageProofsForGame:game.ID];
    return game;
}

-(BOOL)insertNewGamedata:(Gameboard *)game
{
    NSString *insertSQL = [NSString stringWithFormat:@"INSERT INTO %@ (%@,%@,%@,%@,%@) VALUES(\"%@\",\"%@\",\"%@\",%lu,%d)" ,
                           gamesTbName,
                           COL_STARTTIME,
                           COL_ENDTIME,
                           COL_LAYOUT,
                           COL_STATE,
                           COL_WON,
                           [Utility dataBaseDateString:game.startTime],
                           [Utility dataBaseDateString:game.endTime],
                           //[Utility mediumDateString:game.startTime],
                           //[Utility mediumDateString:game.endTime],
                           [game.boardLayout componentsJoinedByString:COMMA_SEPARATOR],
                           game.boardStatusAsNumber,
                           game.bingoVerified];
    
    const char *insert_stmt = [insertSQL UTF8String];
    return [self executeSQLStatement:insert_stmt];
}

-(BOOL)updateGameData:(Gameboard *)game
{
    NSString *updateSQL = [NSString stringWithFormat:@"UPDATE %@ SET %@=\"%@\",%@=\"%@\",%@=\"%@\",%@=%lu,%@=%d WHERE ID=%@" ,
                           gamesTbName,
                           COL_STARTTIME, [Utility dataBaseDateString:game.startTime],
                           COL_ENDTIME, [Utility dataBaseDateString:game.endTime],
                           //COL_STARTTIME, [Utility mediumDateString:game.startTime],
                           //COL_ENDTIME, [Utility mediumDateString:game.endTime],
                           COL_LAYOUT, [game.boardLayout componentsJoinedByString:COMMA_SEPARATOR],
                           COL_STATE, game.boardStatusAsNumber,
                           COL_WON, game.bingoVerified, game.ID];
    
    const char *update_stmt = [updateSQL UTF8String];
    return [self executeSQLStatement:update_stmt];
}

-(BOOL)saveGameData:(Gameboard *)game
{
    if(game.ID) {
        return [self updateGameData:game];
    }
    else{
        return [self insertNewGamedata:game];
    }
    //NSInteger lastRowId = sqlite3_last_insert_rowid(yourdatabasename);
}


//Image Proof related operations
-(NSMutableDictionary *)getImageProofsForGame:(NSNumber *)gameID{
    NSMutableDictionary *_imageProofsDictionary = [[NSMutableDictionary alloc] init];
    if(self.database){
        NSString *querySQL = [NSString stringWithFormat:@"SELECT * FROM  %@ WHERE %@=%@", imageProofsTbName, COL_GAME_ID, gameID];
        const char *query_stmt = [querySQL UTF8String];
        sqlite3_stmt *compiled_sql;
        if(sqlite3_prepare_v2(self.database, query_stmt, -1, &compiled_sql, NULL) == SQLITE_OK){
            while(sqlite3_step(compiled_sql) == SQLITE_ROW){
                
                NSNumber *ID = [NSNumber numberWithUnsignedLong:sqlite3_column_int64(compiled_sql, 0)];
                //1st column is game id
                
                NSString *_dateTakenStr = [[NSString alloc] initWithUTF8String:(const char *)sqlite3_column_text(compiled_sql, 2)];
                NSDate *_dateTaken = [Utility dateFromDatabaseString:_dateTakenStr];
                
                NSString *_signIDStr = [[NSString alloc] initWithUTF8String:(const char *)sqlite3_column_text(compiled_sql, 3)];
                
                NSNumber *_longitude = [NSNumber numberWithDouble:sqlite3_column_double(compiled_sql, 4)];
                
                NSNumber *_latitude = [NSNumber numberWithDouble:sqlite3_column_double(compiled_sql, 5)];
                
                NSString *_imagefilePath = [[NSString alloc] initWithUTF8String:(const char *)sqlite3_column_text(compiled_sql, 6)];
                
                bool _verified = sqlite3_column_int64(compiled_sql, 7);
                
                ImageProof *imageProof = [[ImageProof alloc] initWithExistingImageProofWithFileName:_imagefilePath latitude:_latitude longitude:_longitude checked:_verified bingoSignID:_signIDStr forGame:gameID dateTaken:_dateTaken RecordID:ID];
                [_imageProofsDictionary setObject:imageProof forKey:_signIDStr];
            }
        }
        sqlite3_finalize(compiled_sql);
        sqlite3_close(self.database);
        _database = NULL;
    }
    return _imageProofsDictionary;
}

-(BOOL)saveImageProofData:(ImageProof *)imageProof ForGame:(NSNumber *)gameID
{
    if(imageProof.ID) {
        return [self updateImageProofData:imageProof ForGame:gameID];
    }
    else{
        return [self insertImageProofData:imageProof ForGame:gameID];
    }
}

-(BOOL)insertImageProofData:(ImageProof *)imageProof ForGame:(NSNumber *) gameID
{
    NSString *insertSQL = [NSString stringWithFormat:@"INSERT INTO %@ (%@,%@,%@,%@,%@,%@,%@) VALUES(%@,\"%@\",\"%@\",%@,%@,\"%@\",%@)" ,
                           imageProofsTbName,
                           COL_GAME_ID, COL_DATE_TAKEN, COL_SIGNID,
                           COL_LONGITUDE, COL_LATITUDE, COL_FILEPATH,
                           COL_VERIFIED,
                           gameID, [Utility dataBaseDateString:imageProof.dateTaken], imageProof.bingoSignID,
                           //gameID, [Utility mediumDateString:imageProof.dateTaken], imageProof.bingoSignID,
                           imageProof.longitude, imageProof.latitude, imageProof.fileName, [NSNumber numberWithBool:imageProof.checked]];
    

    const char *insert_stmt = [insertSQL UTF8String];
    return [self executeSQLStatement:insert_stmt];
}

-(BOOL)updateImageProofData:(ImageProof *)imageProof ForGame:(NSNumber *)gameID
{
    NSString *updateSQL = [NSString stringWithFormat:@"UPDATE %@ SET %@=%@,%@=\"%@\",%@=\"%@\",%@=%@,%@=%@,%@=\"%@\",%@=%@ WHERE ID=%@" ,
                           imageProofsTbName,
                           COL_GAME_ID, gameID,
                           COL_DATE_TAKEN, [Utility dataBaseDateString:imageProof.dateTaken],
                           //COL_DATE_TAKEN, [Utility mediumDateString:imageProof.dateTaken],
                           COL_SIGNID, imageProof.bingoSignID,
                           COL_LONGITUDE, imageProof.longitude,
                           COL_LATITUDE, imageProof.latitude,
                           COL_FILEPATH, imageProof.fileName,
                           COL_VERIFIED, [NSNumber numberWithBool:imageProof.checked],
                           imageProof.ID];
    
    const char *update_stmt = [updateSQL UTF8String];
    
    return [self executeSQLStatement:update_stmt];
}

-(NSNumber *)maxRowIDForImageProof{
    return [self maxRowIDForTable:imageProofsTbName];
}


//Game History related operations

-(GameHistory *)getGameHistory
{
    //get start time of last game
    NSString *querySQL = [NSString stringWithFormat:@"SELECT julianday('now','localtime') - julianday(%@) as DURATION1, julianday(%@) - julianday(%@) as DURATION2, %@ FROM %@ ORDER BY ID DESC LIMIT 1", COL_STARTTIME, COL_ENDTIME, COL_STARTTIME, COL_WON, gamesTbName];
    const char *query_stmt = [querySQL UTF8String];
    sqlite3_stmt *compiled_sql;
    bool game_won;
    double currentGameDurationInDays;
    NSNumber *currentGameDurationInMinutes;
    
    if(sqlite3_prepare_v2(self.database, query_stmt, -1, &compiled_sql, NULL) == SQLITE_OK){
        if(sqlite3_step(compiled_sql) == SQLITE_ROW){
            //startTimeStr = [NSString stringWithUTF8String:(const char *) sqlite3_column_text(compiled_sql, 0)];
            //startTime = [Utility dateFromDatabaseString:startTimeStr];
            game_won = sqlite3_column_int(compiled_sql, 2);
            if(!game_won){
                currentGameDurationInDays = sqlite3_column_double(compiled_sql, 0);
            }
            else{
                currentGameDurationInDays = sqlite3_column_double(compiled_sql, 1);
            }
            currentGameDurationInMinutes = [NSNumber numberWithDouble:(currentGameDurationInDays * 24.0 * 60.0)];
        }
    }
    sqlite3_finalize(compiled_sql);
    
    querySQL = [NSString stringWithFormat:@"SELECT COUNT(*), %@ FROM %@ WHERE %@ <> \"\" GROUP BY %@", COL_WON, gamesTbName, COL_ENDTIME, COL_WON];
    query_stmt = [querySQL UTF8String];
    NSNumber *wonCount;
    NSNumber *lostCount;
    bool gameWon;
    if(sqlite3_prepare_v2(self.database, query_stmt, -1, &compiled_sql, NULL) == SQLITE_OK){
        while(sqlite3_step(compiled_sql) == SQLITE_ROW){
            gameWon = sqlite3_column_int(compiled_sql, 1);
            if(gameWon){
                wonCount = [NSNumber numberWithLongLong:sqlite3_column_int64(compiled_sql, 0)];
            }
            else{
                lostCount = [NSNumber numberWithLongLong:sqlite3_column_int64(compiled_sql, 0)];
            }
        }
    }
    sqlite3_finalize(compiled_sql);
    
    querySQL = [NSString stringWithFormat:@"SELECT julianday(%@) - julianday(%@) AS DURATION FROM %@ WHERE %@ <> \"\" AND %@> 0 ORDER BY DURATION ASC LIMIT 1", COL_ENDTIME, COL_STARTTIME, gamesTbName, COL_ENDTIME, COL_WON];
    query_stmt = [querySQL UTF8String];
    NSNumber *fastestTimeMinutes;
    double fastestTimeInDays;
    if(sqlite3_prepare_v2(self.database, query_stmt, -1, &compiled_sql, NULL) == SQLITE_OK){
        if(sqlite3_step(compiled_sql) == SQLITE_ROW){
            fastestTimeInDays = sqlite3_column_double(compiled_sql, 0);
            fastestTimeMinutes = [NSNumber numberWithDouble:(fastestTimeInDays * 24.0 * 60.0)];
        }
    }
    
    GameHistory *history = [[GameHistory alloc] initWithGameDurationMin:currentGameDurationInMinutes numberOfWon:wonCount numberOfLost:lostCount fastestTimeMin:fastestTimeMinutes];
    sqlite3_finalize(compiled_sql);
    sqlite3_close(self.database);
    _database = NULL;
    
    return history;
}


@end
