//
//  Gameboard.m
//  BingoGame
//
//  Created by Jenny Wu on 13/02/14.
//  Copyright (c) 2014 Jenny Wu. All rights reserved.
//

#import "Gameboard.h"
#import "BingoAppDelegate.h"
//#import "BingoDatabase.h"

@interface Gameboard()
@property (nonatomic,strong) Deck *deckOfCards;
@property (nonatomic,readwrite) NSInteger rowSize;
@property (nonatomic,readwrite) NSMutableArray *boardStatus;
@property (nonatomic,readwrite) BOOL bingo;
//@property (nonatomic, readwrite) CFMutableBitVectorRef status;
@property (nonatomic,readwrite) NSMutableArray *winningPatterns;
//the patern that caused bingo
@property (nonatomic,readwrite) unsigned long int bingoPattern;
@property (nonatomic,strong,readwrite) NSDate *startTime;
@property (nonatomic,strong,readwrite) NSDate *endTime;
@property (nonatomic,strong,readwrite) NSNumber *ID;
@property (nonatomic,strong) BingoDatabase *database;
@end


@implementation Gameboard{

}

typedef NS_ENUM(unsigned int, CellStatus){
    CellStatusNotSet = 0,
    CellStatusSet = 1,
};

//@synthesize bingo = _bingo;

//Designated initializer
-(id)initNewGameWithRowSize:(NSInteger)size
{
    self = [super init];
    if (self){
        self.rowSize = size;
        self.boardLayout = [self getNewLayout];
        self.startTime = [NSDate date];
        self.endTime = nil;
        self.bingo = false;
    }
    return self;
}

+(Gameboard *)getLastGame
{
    //return[[[BingoDatabase alloc]init] getLastGame];
    BingoAppDelegate *appDelegate = (BingoAppDelegate *)[[UIApplication sharedApplication] delegate];
    return [appDelegate.bingodatabase getLastGame];
}

-(id)initExistingGameWithLayout:(NSArray *)layout status:(NSNumber *)statusAsNumber StartTime:(NSDate *)start EndTime:(NSDate *)end ID:(NSNumber *)ID;
{
    self = [super init];
    if(self){
        self.rowSize = sqrt([layout count]);
        self.boardLayout = layout;
        self.boardStatus = [self convertUnSignedLongToBoardStataus:statusAsNumber];
        self.bingo = false;
        self.startTime = start;
        self.endTime = end;
        self.ID = ID;
    }
    return self;
}

-(BingoDatabase *)database
{
    if(!_database){
        //_database = [[BingoDatabase alloc] init];
        BingoAppDelegate *appDelegate = (BingoAppDelegate *)[[UIApplication sharedApplication] delegate];
        _database = appDelegate.bingodatabase;
    }
    return _database;
}

-(Deck *)deckOfCards
{
    if(!_deckOfCards){
        _deckOfCards = [[Deck alloc] init];
    }
    return _deckOfCards;
}

-(NSArray *) getNewLayout
{
    return [self.deckOfCards drawRandomCardsWithCount:self.boardSize];
}

-(NSInteger)boardSize
{
    return self.rowSize * self.rowSize;
}

-(NSMutableArray *)boardStatus
{
    if(!_boardStatus){
        _boardStatus = [[NSMutableArray alloc] initWithCapacity:self.boardSize];
        for(int i = 0; i < self.boardSize; i++){
            _boardStatus[i] = [NSNumber numberWithUnsignedInt:CellStatusNotSet];
        }
    }
    return _boardStatus;
}

-(NSMutableDictionary *)dictionaryImageProofs
{
    if(!_dictionaryImageProofs){
        _dictionaryImageProofs = [[NSMutableDictionary alloc] initWithCapacity:self.boardSize];
    }
    return _dictionaryImageProofs;
}

-(bool)isCellSelected:(int)cellIndex{
    return [(NSNumber *)self.boardStatus[cellIndex] unsignedIntValue] == CellStatusSet;
}

//a cell on the game board has been matched
-(void)selectCell:(int)cellIndex
{
    [self setCellwithIndex:cellIndex withCellStatus:CellStatusSet];
    if(self.bingo){
        [_delegate bingoReady:YES];
    }
}

//a cell on the game board no longer has a match
-(void)deselectCell:(int)cellIndex
{
    [self setCellwithIndex:cellIndex withCellStatus:CellStatusNotSet];
    //deselect a cell could cause previous "bingo" status to be invalid.
    self.bingo = NO;
    [_delegate bingoReady:NO];
}

-(void) setCellwithIndex:(int)cellIndex withCellStatus:(CellStatus)status
{
    if ((cellIndex >= 0) && (cellIndex < self.boardSize)){
        self.boardStatus[cellIndex] = [NSNumber numberWithUnsignedInt:status];
        [self saveGameBoard];
    }
}

-(BOOL)bingo
{
    //no need to process again if a bingo pattern was previously found
    if(!_bingo){
        _bingo = false;
        _bingoPattern = 0;
        //check to see if horizontal winning pattern exists
        unsigned long int statusValue = [self convertBoardStatusToUnSignedLong];
        unsigned long int curWinningPattern;
        for(int i=0; i<self.winningPatterns.count; i++){
            curWinningPattern = [(NSNumber *)self.winningPatterns[i] unsignedLongValue];
            if((statusValue & curWinningPattern) == curWinningPattern){
                _bingo = true;
                break;
            }
        }
        if (_bingo) _bingoPattern = curWinningPattern;
    }
    return _bingo;
}

-(BOOL)bingoVerified
{
    BOOL result = NO;
    if(self.bingo){
        result = YES;
        for (int i=0; i<[self.bingoCards count]; i++) {
            NSString *cardID = self.bingoCards[i];
            ImageProof *proof = [self.dictionaryImageProofs objectForKey:cardID];
            if (!proof.checked){
                result = NO;
                break;
            }
        }
    }
    if(result && !self.endTime){
        self.endTime = [NSDate date];
    }
    return result;
}

-(unsigned long int)boardStatusAsNumber
{
    return [self convertBoardStatusToUnSignedLong];
}

//convert the BoardStatus array into one single number, this is done so that bitwise operation can be done to match against the winning pattern
-(unsigned long int) convertBoardStatusToUnSignedLong
{
    unsigned long int result = 0;
    for(int i=0; i < self.boardSize; i++){
        result += [self.boardStatus[i] unsignedIntValue] * pow(2,i);
    }
    return result;
}
         
-(NSMutableArray *) convertUnSignedLongToBoardStataus:(NSNumber *)statusAsNumber
{
    NSMutableArray *status = [[NSMutableArray alloc] initWithCapacity:self.boardSize];
    NSNumber *one = [NSNumber numberWithInt:1];
    NSNumber *zero = [NSNumber numberWithInt:0];
    for(int i = 0; i < self.boardSize; i++){
        status[i] = ([statusAsNumber longValue] & (1<<i)) ? one : zero;
    }
    return status;
}

-(NSMutableArray *)winningPatterns{
    if(!_winningPatterns){
        _winningPatterns = [[NSMutableArray alloc] init];
        [self addHorizontalWinningPattern];
        [self addVerticalWinningPattern];
        [self addDiagonalWinningPattern];
    }
    return _winningPatterns;
}

-(void) addHorizontalWinningPattern{
    //originalHorizontalWinningPattern: 0000000000000000000011111 (for a 5x5 board)
    unsigned long originalHorizontalWinningPattern=0;
    for(int i=0; i<self.rowSize; i++){
        originalHorizontalWinningPattern += pow(2,i);
    }
    
    unsigned long horizontalWinningPattern;
    /* the following for loop is going to generate the following horizontal winning pattern
     0000000000000000000011111  1st row all selected
     0000000000000001111100000  2nd row all selected
     0000000000111110000000000  3rd row all selected
     0000011111000000000000000  4th row all selected
     1111100000000000000000000  5th row all selected
     */
    for (int i=0; i<self.rowSize; i++){
        horizontalWinningPattern = originalHorizontalWinningPattern << (self.rowSize*i);
        [_winningPatterns addObject:[NSNumber numberWithUnsignedLong:horizontalWinningPattern]];
    }
}

-(void) addVerticalWinningPattern{
    //originalVerticalWinningPattern: 0000100001000010000100001 (for a 5x5 board)
    unsigned long originalVerticalWinningPattern = 0;
    //number of rows is of the same value as rowSize.
    for(int i=0; i<self.rowSize; i++){
        originalVerticalWinningPattern += pow(2,i*self.rowSize);
    }
    
    //number of rows is of the same value as rowSize.
    unsigned long verticalWinningPattern;
    /* the following for loop is going to generate the following horizontal winning pattern
     0000100001000010000100001  1st cell of all rows are selected
     0001000010000100001000010  2nd cell of all rows are selected
     0010000100001000010000100  3rd cell of all rows are selected
     0100001000010000100001000  4th cell of all rows are selected
     1000010000100001000010000  5th cell of all rows are selected
     */
    for (int i=0; i<self.rowSize; i++){
        verticalWinningPattern = originalVerticalWinningPattern << i;
        [_winningPatterns addObject:[NSNumber numberWithUnsignedLong:verticalWinningPattern]];
    }
}

-(void) addDiagonalWinningPattern{
    
    unsigned long originalDiagonalWinningPattern = 1;
    //number of rows is of the same value as rowSize. The following for loop will generate DiagonalWinningPattern: 10000,01000,00100,00010,00001 (for a 5x5 board)
    unsigned long diagonalWinningPattern = 0;
    unsigned long shiftLeftBy = 0;
    for(int i=0; i<self.rowSize; i++){
        shiftLeftBy = (self.rowSize+1)*i;
        diagonalWinningPattern += originalDiagonalWinningPattern<<(shiftLeftBy);
        //diagonalWinningPattern += originalDiagonalWinningPattern<<(i*self.rowSize+i);
    }
    [_winningPatterns addObject:[NSNumber numberWithUnsignedLong:diagonalWinningPattern]];
    
    //number of rows is of the same value as rowSize. The following for loop will generate DiagonalWinningPattern: 00001,00010,00100,01000,10000 (for a 5x5 board)
    shiftLeftBy = 0;
    diagonalWinningPattern=0;
    for(int i=1; i<=self.rowSize; i++){
        shiftLeftBy = (self.rowSize-1)*i;
        diagonalWinningPattern += originalDiagonalWinningPattern<<(shiftLeftBy);
    }
    [_winningPatterns addObject:[NSNumber numberWithUnsignedLong:diagonalWinningPattern]];
}


-(NSArray *)bingoCards{
    unsigned long int cellId = 0;
    NSMutableArray *result = [[NSMutableArray alloc] init];
    for(int i=0; i<self.boardSize; i++){
        cellId = pow(2,i); // converting index to binary representation as bingoPattern is in binary form
        if ((self.bingoPattern & cellId) == cellId){
            [result addObject: self.boardLayout[i]];
        }
    }
    return result;
}


-(void)matchCell:(NSInteger)cellIndex withPhotoFile:(NSString *)fileName
{
    ImageProof *imageProof = [self imageProofForCellIndex:cellIndex];
    if(imageProof){
        imageProof.fileName = fileName; //update existing data
        imageProof.dateTaken = [NSDate date];
        imageProof.latitude = nil; //clear previously stored location information
        imageProof.longitude = nil;
    }else{
        NSString *bingoSignID = [self bingoSignIDForCell:cellIndex];
        imageProof = [[ImageProof alloc] initWithFileName:fileName latitude:nil longitude:nil checked:NO bingoSignID:bingoSignID forGame:self.ID];
        [self.dictionaryImageProofs setObject:imageProof forKey:bingoSignID];
    }
    [imageProof saveImageProof];
    [self selectCell:cellIndex];
}


-(void)updateMatchedCell:(NSInteger)cellIndex withLatitude:(NSNumber *)latitude AndLongitude:(NSNumber *)longitude
{
    ImageProof *imageProof = [self imageProofForCellIndex:cellIndex];
    if(imageProof){
        imageProof.latitude = latitude; //clear previously stored location information
        imageProof.longitude = longitude;
    }
    [imageProof saveImageProof];
}

-(ImageProof *)imageProofForCellIndex:(NSInteger)cellIndex
{
    ImageProof *imageProof;
    imageProof = [self.dictionaryImageProofs objectForKey:[self bingoSignIDForCell:cellIndex]];
    return imageProof;
}

-(NSString*) bingoSignIDForCell:(NSInteger)cellIndex
{
    if ((cellIndex > -1) && (cellIndex < self.boardSize)){
        return self.boardLayout[cellIndex];
    }else{
        return nil;
    }
}

-(NSDictionary *)bingoProofs
{
    return self.dictionaryImageProofs;
}

-(BOOL)saveGameBoard
{
    if([self.database saveGameData:self]){
        if (!self.ID){
            self.ID =[self.database maxRowIDForGameBoard];
        }
        return YES;
    }
    else{
        return NO;
    }
}

-(void)quitGame{
    self.endTime = [NSDate date];
    [self saveGameBoard];
}




@end
