//
//  GameboardCollectionViewController.m
//  BingoGame
//
//  Created by Jenny Wu on 13/02/14.
//  Copyright (c) 2014 Jenny Wu. All rights reserved.
//

#import "GameboardCollectionViewController.h"
#import "BingoDetailViewController.h"
#import "Utility.h"
#import "BingoAppDelegate.h"

const int SECTION_COUNT = 1;

@interface GameboardCollectionViewController ()
    @property (nonatomic,strong)Gameboard *gameboard;
@end

@implementation GameboardCollectionViewController{
    Camera *_camera;
    Location *_location;
    BingoAppDelegate *_appdelegate;
    NSInteger _selectedCellId;
}

-(Gameboard *)gameboard{
    return _appdelegate.currentGameBoard;
}

- (id)initWithNibName:(NSString *)nibNameOrNil bundle:(NSBundle *)nibBundleOrNil
{
    self = [super initWithNibName:nibNameOrNil bundle:nibBundleOrNil];
    if (self) {
        // Custom initialization
    }
    return self;
}

- (void)viewDidLoad
{
    [super viewDidLoad];
	// Do any additional setup after loading theview.

    _camera = [[Camera alloc] initWithViewController:self];
    _camera.delegate = self;
    
    _location = [[Location alloc] init];
    _location.delegate = self;

    //set current game board information
    _appdelegate = (BingoAppDelegate *)[[UIApplication sharedApplication] delegate];
    if(self.gameboard.bingo){
        [self bingoReady:YES];
    }
    
    //init selected cell id to -1 to indicate nothing is selected
    _selectedCellId = -1;

}

-(void)alertView:(UIAlertView *)alertView clickedButtonAtIndex:(NSInteger)buttonIndex{
    if (buttonIndex == 1){
        [_appdelegate startNewGame];
        [self refreshScreen];
    }
}

- (void)didReceiveMemoryWarning
{
    [super didReceiveMemoryWarning];
    // Dispose of any resources that can be recreated.
}

-(NSInteger)numberOfSectionsInCollectionView:(UICollectionView *)collectionView
{
    //the number of selection to be displayed on screen, for this app cells belongs to the same section
    return SECTION_COUNT;
}

-(NSInteger)collectionView:(UICollectionView *)collectionView numberOfItemsInSection:(NSInteger)section
{
    return self.gameboard.boardSize;
}

-(UICollectionViewCell *)collectionView:(UICollectionView *)collectionView cellForItemAtIndexPath:(NSIndexPath *)indexPath
{
    GameBoardViewCell *currentCell = [collectionView dequeueReusableCellWithReuseIdentifier:@"GameCell"
                                      forIndexPath:indexPath];
    
    UIImage *image;
    long row = indexPath.row;
    //if (indexPath.section == GAME_CELLS_SECTION_ID){
       
        //NSString *imageFileName = [[_gameboard boardLayout][row] stringValue];
        NSString *imageFileName = [self.gameboard boardLayout][row];
        image = [UIImage imageNamed:imageFileName];
        if ([self.gameboard isCellSelected:row]){
            
            currentCell.imageView.layer.sublayers = nil;
            
            CAShapeLayer *coloredLayer = [self generateColoredCircleLayer:[UIColor greenColor] withinFrame:currentCell.imageView.frame];
            [currentCell.imageView.layer addSublayer: coloredLayer];
        }
        else{
            //currentCell.layer.borderColor = nil;
            //currentCell.layer.borderWidth = 0;
            currentCell.imageView.alpha = 1;
            currentCell.imageView.layer.sublayers = nil;
        }
    
    float radius = currentCell.bounds.size.height * 0.1;
    currentCell.layer.cornerRadius = radius;
    currentCell.layer.masksToBounds = YES;
    
    
    currentCell.imageView.image = image;

    return currentCell;
}

-(CAShapeLayer *) generateColoredCircleLayer: (UIColor *)color withinFrame:(CGRect)frame{
    
    CAShapeLayer *circleLayer = [[CAShapeLayer alloc] init];//
    float offset = frame.size.width * 0.2;
    //float offset = 10;
    //CGPoint position = CGPointMake(-5, -3);
    CGRect rect = CGRectInset(frame, offset, offset);
    circleLayer.fillColor = color.CGColor;
    circleLayer.opacity = 0.9;
    circleLayer.path = [UIBezierPath bezierPathWithOvalInRect:rect].CGPath;
    //circleLayer.position = position;
    //circleLayer.strokeColor = [UIColor magentaColor].CGColor;
    //circleLayer.lineWidth = frame.size.width * 0.04;

    return circleLayer;
}

-(void)viewWillAppear:(BOOL)animated{
    //gameboard might be udpated, so make sure to set the delegate
    self.gameboard.delegate = self; //collection view will be notified if game state turns into BINGO

    [self refreshScreen];
    
    if(_appdelegate.currentGameBoard.bingoVerified){
        UIAlertView *alert = [[UIAlertView alloc] initWithTitle: @"Bingo" message:@"You have won the current game. Do you want to start a new game?" delegate:self cancelButtonTitle:@"No" otherButtonTitles:@"Yes", nil];
        [alert show];
    }
}

-(void)refreshScreen{
    [self.collectionView reloadData];
    self.collectionView.userInteractionEnabled = self.gameboard.bingoVerified? NO: YES;
    self.collectionView.alpha = self.gameboard.bingoVerified? 0.7 : 1;
}

//private function to create a button inside a cell at run time
-(UIButton *)createButtonForCell:(UICollectionViewCell *)parentCell
{
    UIButton *button = [UIButton buttonWithType: UIButtonTypeSystem];
    CGRect rect = CGRectMake(0, 0, parentCell.frame.size.width, parentCell.frame.size.height);
    button.frame = rect;
    [parentCell addSubview:button];
    return button;
}


-(void)collectionView:(UICollectionView *)collectionView didSelectItemAtIndexPath:(NSIndexPath *)indexPath
{
    //if a cell has an image assigned to it already, show preview window which allows user to retake photo
    _selectedCellId = indexPath.row;
    if([self.gameboard isCellSelected:_selectedCellId]){
        [self loadImageProofPreview];
    }
    else{
        [_camera takePhotoForCell:_selectedCellId];
    }
}

-(void)loadImageProofPreview{
    //show image proof preview screen
    [self performSegueWithIdentifier:@"imageProofPreview" sender:self];
}

-(void)prepareForSegue:(UIStoryboardSegue *)segue sender:(id)sender{
    //assign the necessary information to BingoDetailViewController aka image proof preview window.
    //The same scene used to display image when user hit the bingo button.
    if ([segue.identifier isEqual: @"imageProofPreview"]){
        NSIndexPath *selecteCellPath = [self currentSelectedCell];
        if (selecteCellPath) {
            BingoDetailViewController * imagePreviewController = [segue destinationViewController];
            imagePreviewController.retakePhotoDelegate = self;
            imagePreviewController.imageProof = [self.gameboard imageProofForCellIndex:selecteCellPath.row];
            imagePreviewController.isEditMode = YES;
        }
    }
}

-(void)retakeImageProofPhoto{
    //this is a delegate method being called by the BingoDetailViewController
    //after user tab on the "retake" button and after the BingoDetailViewController has been dismissed completely
    //otherwise, imagepicker somehow does not like to be displayed.
    [_camera takePhotoForCell:_selectedCellId];
    _selectedCellId = -1;
}

-(void)btnCameraClicked:(UIButton *) button
{
    [_camera takePhotoForCell:_selectedCellId];
}

-(NSIndexPath *) currentSelectedCell{
    NSArray *indexPaths = [self.collectionView indexPathsForSelectedItems];
    if ([indexPaths count] > 0) {
        return indexPaths[0];
    }
    return nil;
}

-(void)cameraDidTakePhoto:(NSString *)fileName ForCell:(NSInteger)cellID{
    //NSIndexPath *selecteCellPath = [self currentSelectedCell];
    
    if (cellID>-1 && cellID<self.gameboard.boardSize) {
        //NSInteger selectedCellRow = [selecteCellPath row];
        [self.gameboard matchCell:cellID withPhotoFile:fileName];
        
        [_location startUpdatingLocationForBingoCell:cellID];

        [self.collectionView reloadItemsAtIndexPaths: self.collectionView.indexPathsForSelectedItems];

        //NSString * msg = [NSString stringWithFormat:@"%d",(int)row];
        
        //[Utility showAlertMessageWithTitle:@"Selected cell" message:msg delegate:nil canceltButtonText:@"ok"];
    }
    //_gameboard matchCell:<#(int)#> withPhotoFile:fileName;
}

-(void)cameraFailToTakePhoto{
    //controller does not need to do anything
}

-(void)locationManagerDidUpdateLatitude:(NSNumber *)latitude longtitude:(NSNumber *)longitude forCell:(NSInteger)cellRowID
{
    [self.gameboard updateMatchedCell:cellRowID withLatitude:latitude AndLongitude:longitude];
    //NSString *msg = [NSString stringWithFormat:@"%@, %@, for cell:%ld", latitude, longitude,(long)cellRowID];
    //[Utility showAlertMessageWithTitle:@"location" message:msg delegate:nil canceltButtonText:@"ok"];
}


//This method will be called if locataion service failed to get current location, this includes the case where location service is disabled.
-(void)locationManagerUpdateFailedWithError:(NSError *)error
{
    //contoller does not need to do anything, current game rule: as long as a photo file is saved that's enough. Location information is nice to have.
}

-(void)bingoReady:(bool)ready{
    BingoAppDelegate *delegate =(BingoAppDelegate *)[[UIApplication sharedApplication] delegate];
    UITabBarItem *bingoTabbarItem = ((UIViewController *)[self.tabBarController.viewControllers objectAtIndex:delegate.bingoTabItemIndex]).tabBarItem;
    if(ready){
        bingoTabbarItem.badgeValue = @"Ready";
    }
    else{
        bingoTabbarItem.badgeValue = nil;
    }
}





@end
