//
//  MainViewController.m
//  holeinsome
//
//  Created by Antti Myllykoski on 2/19/14.
//  Copyright (c) 2014 amyllykoski.com. All rights reserved.
//
#import <QuartzCore/QuartzCore.h>
#import "MainViewController.h"
#import "HorizontalPickerView.h"
#import "PlayersViewController.h"
#import "CoursesViewController.h"
#import "CourseViewController.h"
#import "PlayerViewController.h"
#import "ScoreViewController.h"
#import "Toast+UIView.h"
#import "ScoreCell.h"
#import "PlayerDAO.h"
#import "TeeColorDAO.h"
#import "HoleDAO.h"
#import "CourseDAO.h"

@interface MainViewController ()
@property (weak, nonatomic) NSArray * source;
@property (strong, nonatomic) NSArray * dataSource;
@property NSMutableArray * players;
@property PlayerDAO * playerDAO;
@property TeeColorDAO * teeColorDAO;
@property HoleDAO * holeDAO;
@property CourseDAO * courseDAO;
@property NSMutableArray * teeColors;
@property Course * currentCourse;
@end


static int currentHole = 1;

@implementation MainViewController

static int highestHolePlayed;

+ (int) highestHolePlayed
{
    return highestHolePlayed;
}

NSMutableArray * holeNumbers;

- (void) viewDidLoad
{
    [super viewDidLoad];
    [self initDAOs];
    [self initUIComponents];
}

- (void) initDAOs
{
    if (_playerDAO == nil)
    {
        _playerDAO = [[PlayerDAO alloc] init];
        [_playerDAO initDb];
    }
    _players = [_playerDAO getPlayersOnScorecard];
    
    if (_teeColorDAO == nil)
    {
        _teeColorDAO = [[TeeColorDAO alloc] init];
        [_teeColorDAO initDb];
    }
    _teeColors = [_teeColorDAO getTeeColors];
    [_teeColors insertObject:[[TeeColor alloc] init] atIndex:0];
    
    if (_holeDAO == nil)
    {
        _holeDAO = [[HoleDAO alloc] init];
        [_holeDAO initDb];
    }
    
    if (_courseDAO == nil)
    {
        _courseDAO = [[CourseDAO alloc] init];
        [_courseDAO initDb];
    }
}

- (void) initUIComponents
{
    _source = self.dataSource;
    //    if(_currentCourse == nil) {
    //        currentHole = highestHolePlayed = 1;
    //    }
    
    [[UITableViewHeaderFooterView appearance] setTintColor:[UIColor clearColor]];
    
    self.navigationController.toolbar.backgroundColor = [UIColor clearColor];
    self.view.backgroundColor = [UIColor colorWithPatternImage:[UIImage imageNamed:@"background.png"]];
    
    UISwipeGestureRecognizer * rightRecognizer = [[UISwipeGestureRecognizer alloc] initWithTarget:self action:@selector(rightSwipeHandler:)];
    [rightRecognizer setDirection:(UISwipeGestureRecognizerDirectionRight)];
    [self.view addGestureRecognizer:rightRecognizer];
    UISwipeGestureRecognizer * leftRecognizer = [[UISwipeGestureRecognizer alloc] initWithTarget:self action:@selector(leftSwipeHandler:)];
    [leftRecognizer setDirection:(UISwipeGestureRecognizerDirectionLeft)];
    [_swipeView addGestureRecognizer:leftRecognizer];
}

- (void) viewWillAppear:(BOOL)animated
{
    [self.navigationController setNavigationBarHidden:YES];
    _players = [_playerDAO getPlayersOnScorecard];
    [_tableView reloadData];
    _currentCourse = [_courseDAO getSelectedCourse];
    if (_currentCourse)
    {
        _courseNameLabel.text = _currentCourse.name;
        [_scoreCardButton setEnabled:YES];
    }
    else
    {
        currentHole = highestHolePlayed = 1;
        [_scoreCardButton setEnabled:NO];
    }
    [self updateTeeColorInfo];
}

NSString * (^ conv)(NSInteger val) = ^(NSInteger val)
{
    return [NSString stringWithFormat:@"%ld", (long) val];
};

- (void) scrollLeft
{
    CATransition * animation = [CATransition animation];
    
    [animation setDelegate:self];
    [animation setType:kCATransitionPush];
    [animation setSubtype:kCATransitionFromLeft];
    [animation setDuration:0.40];
    [animation setTimingFunction:
     [CAMediaTimingFunction functionWithName:kCAMediaTimingFunctionEaseInEaseOut]];
    [_swipeView.layer addAnimation:animation forKey:kCATransition];
}

- (void) rightSwipeHandler:(UISwipeGestureRecognizer *)recognizer
{
    [self updatePlayersInDB];
    if (currentHole > 1)
    {
        currentHole--;
        [self scrollLeft];
        _currentHoleField.text = conv(currentHole);
        [self updateTeeColorInfo];
        [_tableView reloadData];
    }
}

- (void) scrollRight
{
    CATransition * animation = [CATransition animation];
    
    [animation setDelegate:self];
    [animation setType:kCATransitionPush];
    [animation setSubtype:kCATransitionFromRight];
    [animation setDuration:0.50];
    [animation setTimingFunction:
     [CAMediaTimingFunction functionWithName:kCAMediaTimingFunctionEaseInEaseOut]];
    [_swipeView.layer addAnimation:animation forKey:kCATransition];
}

- (void) leftSwipeHandler:(UISwipeGestureRecognizer *)recognizer
{
    [self updatePlayersInDB];
    if (currentHole < 18)
    {
        currentHole++;
        highestHolePlayed = currentHole;
        [self scrollRight];
        _currentHoleField.text = conv(currentHole);
        [self updateTeeColorInfo];
        [_tableView reloadData];
    }
}

- (void) didReceiveMemoryWarning
{
    [super didReceiveMemoryWarning];
}


- (void) viewDidAppear:(BOOL)animated
{
    [super viewDidAppear:animated];
    _players = [_playerDAO getPlayersOnScorecard];
    _currentHoleField.text = conv(currentHole);
    [[self tableView] reloadData];
}

#pragma mark -  HPickerViewDataSource

- (NSInteger) numberOfRowsInPickerView:(HorizontalPickerView *)pickerView
{
    return self.source.count;
}

#pragma mark -  HPickerViewDelegate

- (NSString *) pickerView:(HorizontalPickerView *)pickerView titleForRow:(NSInteger)row
{
    return [NSNumberFormatter localizedStringFromNumber:self.source[row] numberStyle:NSNumberFormatterDecimalStyle];
}

- (void) pickerView:(HorizontalPickerView *)pickerView didSelectRow:(NSInteger)row
{
    //    self.selectedRowLabel.text = [NSString stringWithFormat:@"%@", @(row)];
    int currentHoleBeforeChange = (int) currentHole;
    
    currentHole = (int) row + 1;
    _currentHoleField.text = conv(currentHole);
    NSLog(@"Selected %@", [NSString stringWithFormat:@"Hole %@", @(currentHole)]);
    if (currentHoleBeforeChange > currentHole)
    {
        [self scrollLeft];
    }
    else
    {
        [self scrollRight];
    }
}

- (NSArray *) dataSource
{
    if (nil == _dataSource)
    {
        _dataSource = @[@1, @2, @3, @4, @5, @6, @7, @8, @9, @10, @11, @12, @13, @14, @15, @16, @17, @18];
    }
    return _dataSource;
}

- (NSInteger) tableView:(UITableView *)tableView numberOfRowsInSection:(NSInteger)section
{
    // Make room for "Add Player" -row if the scorecard is not full
    return [_players count] + 1;
}

- (NSInteger) numberOfSectionsInTableView:(UITableView *)tableView
{
    return 1;
}

- (NSString *) tableView:(UITableView *)tableView titleForHeaderInSection:(NSInteger)section
{
    return @" ";
}

- (UITableViewCell *) tableView:(UITableView *)tableView cellForRowAtIndexPath:(NSIndexPath *)indexPath
{
    if ([_players count] <= indexPath.row)
    {
        UITableViewCell * cell = [tableView dequeueReusableCellWithIdentifier:@"addAplayerCell"];
        return cell;
    }
    else
    {
        Player * p = [_players objectAtIndex:indexPath.row];
        ScoreCell * cell = [tableView dequeueReusableCellWithIdentifier:@"scoreCell"];
        cell = [cell init];
        if (p.imagePath.length > 0)
        {
            NSString * imageFilePath = [NSHomeDirectory() stringByAppendingPathComponent:p.imagePath];
            cell.image.image = [UIImage imageWithContentsOfFile:imageFilePath];
        }
        else
        {
            cell.image.image = [UIImage imageNamed:@"golf_hat.png"];
        }
        cell.firstName.text = p.firstName;
        cell.lastName.text = p.lastName;
        [cell.teeColor setBackgroundColor:[[_teeColors objectAtIndex:p.teeColor] getUIColorFromRGB]];
        cell.teeColor.layer.cornerRadius = 14;
        cell.tag = p.playerId;
        [cell.strokePicker selectRow:[[p getStrokesForHole:currentHole] integerValue] - 1 animated:YES];
        return cell;
    }
}


- (BOOL) tableView:(UITableView *)tableView canEditRowAtIndexPath:(NSIndexPath *)indexPath
{
    if ([_players count] > 0 && indexPath.row < [_players count])
    {
        return YES;
    }
    else
    {
        return NO;
    }
}

- (void) tableView:(UITableView *)tableView commitEditingStyle:(UITableViewCellEditingStyle)editingStyle forRowAtIndexPath:(NSIndexPath *)indexPath
{
    Player * p = (Player *) [_players objectAtIndex:indexPath.row];
    
    [p setIsRecent:NO];
    [_playerDAO updatePlayer:p];
    [_players removeObjectAtIndex:indexPath.row];
    if (editingStyle == UITableViewCellEditingStyleDelete)
    {
        [tableView beginUpdates];
        [tableView deleteRowsAtIndexPaths:@[indexPath] withRowAnimation:UITableViewRowAnimationAutomatic];
        [tableView endUpdates];
    }
    [[self tableView]reloadData];
    [self updateTeeColorInfo];
}


- (BOOL) tableView:(UITableView *)tableView canMoveRowAtIndexPath:(NSIndexPath *)indexPath
{
    return YES;
}

- (CGFloat) tableView:(UITableView *)tableView heightForRowAtIndexPath:(NSIndexPath *)indexPath
{
    float heightForRow = 74.;
    
    if (indexPath.row == 4)
    {
        return 0;   // Don't show add player -line when the scorecard is full
    }
    else
    {
        return heightForRow;
    }
}


- (NSString *) tableView:(UITableView *)tableView titleForDeleteConfirmationButtonForRowAtIndexPath:(NSIndexPath *)indexPath
{
    return @"Remove player \nfrom scorecard";
}

- (void) prepareForSegue:(UIStoryboardSegue *)segue sender:(id)sender
{
    [self.navigationController setNavigationBarHidden:NO];
    [self updatePlayersInDB];
    
    if ([[segue identifier] isEqualToString:@"choosePlayer"])
    {
        UINavigationController * navController = (UINavigationController *) [segue destinationViewController];
        PlayersViewController * vc = (PlayersViewController * ) [navController topViewController];
        [vc setIsChoosingPlayer:YES];
    }
    else if ([[segue identifier] isEqualToString:@"showPlayerDetails"])
    {
        PlayerViewController * vc = (PlayerViewController *) [segue destinationViewController];
        [vc setSelectedPlayerId:[sender tag]];
    }
    else if ([[segue identifier] isEqualToString:@"managePlayers"])
    {
        UINavigationController * navController = (UINavigationController *) [segue destinationViewController];
        PlayersViewController * vc = (PlayersViewController * ) [navController topViewController];
        [vc setIsChoosingPlayer:NO];
    }
    else if ([[segue identifier] isEqualToString:@"manageCourses"])
    {
        //    UINavigationController * navController = (UINavigationController *) [segue destinationViewController];
        //    CoursesViewController * vc = (CoursesViewController * ) [navController topViewController];
    }
    else if ([[segue identifier] isEqualToString:@"viewCourse"])
    {
        //    UINavigationController * navController = (UINavigationController *) [segue destinationViewController];
        //    CoursesViewController * vc = (CoursesViewController * ) [navController topViewController];
        CourseViewController * vc = [segue destinationViewController];
        [vc setSelectedCourseId:_currentCourse.courseId];
    }
    else if ([[segue identifier] isEqualToString:@"chooseCourse"])
    {
        //        UINavigationController * navController = (UINavigationController *) [segue destinationViewController];
        //        CoursesViewController * vc = (CoursesViewController * ) [navController topViewController];
    }
    else if ([[segue identifier] isEqualToString:@"courseDetails"])
    {
        //        UINavigationController * navController = (UINavigationController *) [segue destinationViewController];
        //        CourseViewController * vc = (CourseViewController * ) [navController topViewController];
        // [vc setIsChoosingCourse:YES];
    }
    else if ([[segue identifier] isEqualToString:@"showScorecard"])
    {
        UINavigationController * navController = (UINavigationController *) [segue destinationViewController];
        ScoreViewController * vc = (ScoreViewController * ) [navController topViewController];
        [vc setIsFrontNine:YES];
        [vc setGolfCourseName: [_currentCourse name]];
    }
}

- (void) touchesBegan:(NSSet *)touches withEvent:(UIEvent *)event
{
    for (UIView * txt in self.view.subviews)
    {
        if ([txt isKindOfClass:[UITextField class]] && [txt isFirstResponder])
        {
            [txt resignFirstResponder];
        }
    }
}

- (IBAction) onLeftUpperParHcp:(id)sender
{
    NSLog(@"Lefttupperparhcp");
}


- (IBAction) onRightUpperParHcp:(id)sender
{
    NSLog(@"Rightupperparhcp");
}

- (IBAction) onLeftLowerParHcp:(id)sender
{
    NSLog(@"Leftlowerparhcp");
}

- (IBAction) onRightLowerParHcp:(id)sender
{
    NSLog(@"Rightlowerparhcp");
}

- (IBAction) onCourseNameTapped:(id)sender
{
    if (_currentCourse != nil)
    {
        [self performSegueWithIdentifier:@"viewCourse" sender:self];
    }
    else
    {
        [self toast:@"No Golf Course Selected" msg:@"Please Select Golf Course."];
    }
}

- (void) updateTeeColorInfo
{
    NSMutableDictionary * hcpParLookup = [[NSMutableDictionary alloc] init];
    
    for (int i = 0; i < 4; i++)
    {
        [[_parLabel objectAtIndex:i] setHidden:YES];
        [[_hcpLabel objectAtIndex:i] setHidden:YES];
        [[_parText objectAtIndex:i] setHidden:YES];
        [[_hcpText objectAtIndex:i] setHidden:YES];
        [[_teeColorBox objectAtIndex:i] setHidden:YES];
    }
    
    if (_currentCourse == nil)
    {
        return;
    }
    
    Hole * h = nil;
    for (Player * p in _players)
    {
        h = [_holeDAO getHole:(int) _currentCourse.courseId teeColorId:(int) p.teeColor holeNumber:(int) currentHole];
        NSString * key = conv(p.teeColor);
        [hcpParLookup setObject:h forKey:key];
    }
    
    int i = 0;
    for (id key in hcpParLookup)
    {
        id h = [hcpParLookup objectForKey:key];
        ((UITextField *) [_parText objectAtIndex:i]).text = conv( (int) ((Hole *) h).par);
        ((UITextField *) [_hcpText objectAtIndex:i]).text = conv( (int) ((Hole *) h).hcp);
        [((UITextField *) [_teeColorBox objectAtIndex:i])setBackgroundColor :[self getUIColorFor:((Hole *) h).teeColorId]];
        
        [[_parLabel objectAtIndex:i] setHidden:NO];
        [[_hcpLabel objectAtIndex:i] setHidden:NO];
        [[_parText objectAtIndex:i] setHidden:NO];
        [[_hcpText objectAtIndex:i] setHidden:NO];
        [[_teeColorBox objectAtIndex:i] setHidden:NO];
        ((UITextField *) [_teeColorBox objectAtIndex:i]).layer.cornerRadius = 12.5;
        i++;
    }
}

- (UIColor *) getUIColorFor:(NSInteger)teeColorId
{
    for (TeeColor * t in _teeColors)
    {
        if (t.teeColorId == teeColorId)
        {
            return t.getUIColorFromRGB;
        }
    }
    return nil;
}

- (void) updatePlayersInDB
{
    NSArray * cells = [_tableView visibleCells];
    int i = 0;
    NSInteger score = 0;
    
    for (Player * p in _players)
    {
        score = [((ScoreCell *) [cells objectAtIndex:i++]).strokePicker selectedRow];
        [p setStrokes:conv( (int) score + 1) forHole:conv((int) currentHole)];
        [_playerDAO updatePlayer:p];
    }
}

- (void) toast:(NSString *)title msg:(NSString *)msg
{
    [self.view makeToast:msg
                duration:3.0
                position:@"center"
                   title:title];
}
@end
