//
//  PlayerViewController.m
//  holeinsome
//
//  Created by Antti Myllykoski on 1/21/14.
//  Copyright (c) 2014 amyllykoski.com. All rights reserved.
//

#import <MobileCoreServices/MobileCoreServices.h>
#import "PlayerViewController.h"
#import "PlayerDao.h"
#import "TeeColorDAO.h"
#import "Toast+UIView.h"

#define kImageSize 96

@interface PlayerViewController ()
@property NSString * imageFilename;
@property NSInteger selectedTeeColorId;
@property TeeColorDAO * teeColorDAO;
@property NSMutableArray * teeColors;
@end

@implementation PlayerViewController

- (void)initIvars:(Player *)p
{
    _firstName.delegate = self;
    _lastName.delegate = self;
    _email.delegate = self;
    _hdcp.delegate = self;
    _firstName.text = p.firstName;
    _lastName.text = p.lastName;
    _email.text = p.email;
    _hdcp.text = [NSString stringWithFormat:@"%.01f", p.hdcp];
    _imageFilename = p.imagePath;
    _selectedTeeColorId = p.teeColor;
    [_teeColorField setBackgroundColor:[[_teeColors objectAtIndex:_selectedTeeColorId]getUIColorFromRGB]];
    _teeColorField.layer.cornerRadius = 20;
    _teeColorNameLabel.text = [[_teeColors objectAtIndex:_selectedTeeColorId] teeColorName];
    
    NSString * imagePath = [NSHomeDirectory() stringByAppendingPathComponent:p.imagePath];
    _playerImage.image = [UIImage imageWithContentsOfFile:imagePath];
}

- (void) viewDidLoad
{
    [super viewDidLoad];
    
    self.view.backgroundColor = [UIColor colorWithPatternImage:[UIImage imageNamed:@"background.png"]];
    if (_teeColorDAO == nil)
    {
        _teeColorDAO = [[TeeColorDAO alloc] init];
        [_teeColorDAO initDb];
    }
    _teeColors = [_teeColorDAO getTeeColors];
    [_teeColors insertObject:[[TeeColor alloc] init] atIndex:0];
    
    if (_selectedPlayerId == -1)
    {
        _selectedTeeColorId = 1;
        [_teeColorField setBackgroundColor:[[_teeColors objectAtIndex:_selectedTeeColorId] getUIColorFromRGB]];
        _teeColorNameLabel.text = [[_teeColors objectAtIndex:_selectedTeeColorId] teeColorName];
        return;
    }
    
    PlayerDAO * dao = [[PlayerDAO alloc] init];
    [dao initDb];
    Player * p = [dao getPlayer:(int) _selectedPlayerId];
    if (p == nil)
    {
        NSLog(@"Player with id; %i not found.", (int) _selectedPlayerId);
        return;
    }
    
    [self initIvars:p];
    
    [self.navigationController setNavigationBarHidden:NO];
}

- (void) didReceiveMemoryWarning
{
    [super didReceiveMemoryWarning];
}

- (void) touchesBegan:(NSSet *)touches withEvent:(UIEvent *)event
{
    for (UIView * txt in self.view.subviews)
    {
        if ([txt isKindOfClass:[UITextField class]] && [txt isFirstResponder])
        {
            [txt resignFirstResponder];
        }
    }
}

-(BOOL)textFieldShouldReturn:(UITextField*)textField;
{
    NSInteger nextTag = textField.tag + 1;
    UIResponder* nextResponder = [textField.superview viewWithTag:nextTag];
    if (nextResponder) {
        [nextResponder becomeFirstResponder];
    } else {
        [textField resignFirstResponder];
    }
    return NO;
}


- (IBAction) takePicture:(id)sender
{
    [_firstName becomeFirstResponder];
    [_firstName resignFirstResponder];
    
#if TARGET_IPHONE_SIMULATOR
    [self getImageFromAssets];
    _newPic = YES;
    return;
#endif
    
    if ([UIImagePickerController isSourceTypeAvailable:
         UIImagePickerControllerSourceTypeCamera])
    {
        UIImagePickerController * imagePicker = [[UIImagePickerController alloc] init];
        imagePicker.delegate = self;
        imagePicker.sourceType = UIImagePickerControllerSourceTypeCamera;
        imagePicker.mediaTypes = @[(NSString *) kUTTypeImage];
        imagePicker.allowsEditing = NO;
        [self presentViewController:imagePicker animated:YES completion:nil];
        _newPic = YES;
    }
}

- (void) saveOrUpdatePlayer
{
    [self.view endEditing:YES];
    Player * p = [[Player alloc] initWithData:_firstName.text
                                     lastName:_lastName.text
                                    imagePath:_imageFilename
                                        email:_email.text
                                         hdcp:[_hdcp.text doubleValue]
                                     teeColor:_selectedTeeColorId == 0 ? 1:_selectedTeeColorId];
    
    if (!p.imagePath)
    {
        [p setImagePath:@""];
    }
    
    [p setPlayerId:_selectedPlayerId];
    
    PlayerDAO * dao = [[PlayerDAO alloc] init];
    [dao initDb];
    if (_selectedPlayerId == -1)
    {
        if ([dao addPlayer:p])
        {
            [self toast:@"Player Saved" msg:@"Player saved to DB."];
        }
        else
        {
            [self toast:@"Player Save Failed" msg:@"Could not save player to DB."];
        }
    }
    else
    {
        Player * currentPlayer = [dao getPlayer:(int) _selectedPlayerId];
        p.isRecent = currentPlayer.isRecent;
        if ([dao updatePlayer:p])
        {
            [self toast:@"Player Updated" msg:@"Player updated to DB."];
        }
        else
        {
            [self toast:@"Player Update Failed" msg:@"Could not update player to DB."];
        }
    }
}

- (IBAction) saveData:(id)sender
{
    if (_firstName.text.length == 0)
    {
        [self toast:@"Missing First Name" msg:@"Please give first name."];
        return;
    }
    
    //  Relax the information requirements: first name suffices.
    //    if (_lastName.text.length == 0)
    //    {
    //        [self toast:@"Missing Last Name" msg:@"Please give last name."];
    //        return;
    //    }
    //    if (_email.text.length == 0)
    //    {
    //        [self toast:@"Missing Email Address" msg:@"Please give email address."];
    //        return;
    //    }
    
    [self saveOrUpdatePlayer];
    [self.navigationController popViewControllerAnimated:YES];
}

- (IBAction) goBack:(id)sender
{
    [self saveData:self];
    [self.navigationController popViewControllerAnimated:YES];
}

- (IBAction) setTeeColor:(id)sender
{
    UIActionSheet * popup = [[UIActionSheet alloc] initWithTitle:@"Choose Tee Color You Want to Play"
                                                        delegate:self
                                               cancelButtonTitle:nil
                                          destructiveButtonTitle:nil
                                               otherButtonTitles:nil];
    
    for (TeeColor * t in _teeColors)
    {
        [popup addButtonWithTitle:t.teeColorName];
    }
    [popup addButtonWithTitle:@"Cancel"];
    [popup addButtonWithTitle:Nil];
    popup.tag = 1;
    [popup showInView:[UIApplication sharedApplication].keyWindow];
}

- (void) actionSheet:(UIActionSheet *)popup clickedButtonAtIndex:(NSInteger)buttonIndex
{
    switch (popup.tag)
    {
        case 1 :
            if (buttonIndex < [_teeColors count])  // not cancel
            {
                _selectedTeeColorId = buttonIndex;
                [_teeColorField setBackgroundColor:[[_teeColors objectAtIndex:_selectedTeeColorId]getUIColorFromRGB]];
                _teeColorNameLabel.text = [[_teeColors objectAtIndex:_selectedTeeColorId] teeColorName];
            }
            break;
        default:
            break;
    }
}

- (NSString *) currentDateandTime
{
    NSDate * today = [NSDate date];
    NSDateFormatter * dateFormat = [[NSDateFormatter alloc] init];
    
    [dateFormat setDateFormat:@"MMddyyyy_HHmmssSS"];
    NSString * dateString = [dateFormat stringFromDate:today];
    
    return dateString;
}

- (void) imagePickerController:(UIImagePickerController *)picker
 didFinishPickingMediaWithInfo:(NSDictionary *)info
{
    NSString * mediaType = info[UIImagePickerControllerMediaType];
    
    [self dismissViewControllerAnimated:YES completion:nil];
    
    if ([mediaType isEqualToString:(NSString *) kUTTypeImage])
    {
        UIImage * image = info[UIImagePickerControllerOriginalImage];
        
        if (image.size.width != kImageSize || image.size.height != kImageSize)
        {
            CGSize itemSize = CGSizeMake(kImageSize, kImageSize);
            UIGraphicsBeginImageContextWithOptions(itemSize, NO, 0.0f);
            CGRect imageRect = CGRectMake(0.0, 0.0, itemSize.width, itemSize.height);
            [image drawInRect:imageRect];
            _playerImage.image = UIGraphicsGetImageFromCurrentImageContext();
            UIGraphicsEndImageContext();
        }
        
        NSString * date = [self currentDateandTime];
        _imageFilename = [NSString stringWithFormat:@"Documents/%@.png", date];
        NSString * imagePath = [NSHomeDirectory() stringByAppendingPathComponent:_imageFilename];
        [UIImagePNGRepresentation(_playerImage.image) writeToFile:imagePath atomically:YES];
        NSLog(@"Saved image %@.", _imageFilename);
    }
}

- (void) image:(UIImage *)image
finishedSavingWithError:(NSError *)error
   contextInfo:(void *)contextInfo
{
    if (error)
    {
        UIAlertView * alert = [[UIAlertView alloc]
                               initWithTitle:@"Save failed"
                               message:@"Failed to save image"
                               delegate:nil
                               cancelButtonTitle:@"OK"
                               otherButtonTitles:nil];
        [alert show];
    }
}

- (void) imagePickerControllerDidCancel:(UIImagePickerController *)picker
{
    [self dismissViewControllerAnimated:YES completion:nil];
}

- (void) toast:(NSString *)title msg:(NSString *)msg
{
    [self.view makeToast:msg
                duration:3.0
                position:@"center"
                   title:title];
}

/*
 For supporting images when running on simulator.
 */

- (void) getImageFromAssets
{
    NSArray * rndPics = @[@"tiger", @"golfr", @"golfr-lady", @"phil"];
    int rnd = rand() % ([rndPics count]);
    NSString * pic = [NSString stringWithFormat:@"%@", rndPics[rnd]];
    UIImage * image = [UIImage imageNamed:pic];
    
    if (image.size.width != kImageSize || image.size.height != kImageSize)
    {
        CGSize itemSize = CGSizeMake(kImageSize, kImageSize);
        UIGraphicsBeginImageContextWithOptions(itemSize, NO, 0.0f);
        CGRect imageRect = CGRectMake(0.0, 0.0, itemSize.width, itemSize.height);
        [image drawInRect:imageRect];
        _playerImage.image = UIGraphicsGetImageFromCurrentImageContext();
        UIGraphicsEndImageContext();
    }
    
    NSString * date = [self currentDateandTime];
    _imageFilename = [NSString stringWithFormat:@"Documents/%@.png", date];
    NSString * imagePath = [NSHomeDirectory() stringByAppendingPathComponent:_imageFilename];
    [UIImagePNGRepresentation(_playerImage.image) writeToFile:imagePath atomically:YES];
    NSLog(@"Saved image %@.", _imageFilename);
}

@end
