//
//  ScoreViewController.m
//  holeinsome
//
//  Created by Antti Myllykoski on 3/2/14.
//  Copyright (c) 2014 amyllykoski.com. All rights reserved.
//

#import "ScoreViewController.h"
#import "MainViewController.h"
#import "PlayerDAO.h"
#import "CourseDAO.h"
#import "CourseDetailDAO.h"
#import "HoleDAO.h"
#import "TeeColorDAO.h"

@interface ScoreViewController ()
@property PlayerDAO * playerDAO;
@property CourseDAO * courseDAO;
@property CourseDetailDAO * courseDetailDAO;
@property HoleDAO * holeDAO;
@property TeeColorDAO * teeColorDAO;
@property NSMutableArray * players;
@property NSMutableArray * teeColors;
@end

@implementation ScoreViewController

- (void) viewWillAppear:(BOOL)animated
{
    [self.navigationController.navigationBar.layer removeAllAnimations];
    [self buildScorecard];
}

- (void) calculateGrossScores
{
    _player1Gross.text = conV([_player1In.text integerValue] + [_player1Out.text integerValue]);
    _player2Gross.text = conV([_player2In.text integerValue] + [_player2Out.text integerValue]);
    _player3Gross.text = conV([_player3In.text integerValue] + [_player3Out.text integerValue]);
    _player4Gross.text = conV([_player4In.text integerValue] + [_player4Out.text integerValue]);
}

- (void) buildScorecard
{
    self.navigationItem.title = _golfCourseName;
    [self setHoleNumbers];
    if (_isFrontNine)
    {
        [self setParValuesOnFrontNineScorecard:NO];
        [self setParValuesOnBackNineScorecard:YES];
        [self setPlayerScoresOnFrontNineScorecard:NO];
        [self setPlayerScoresOnBackNineScorecard:YES];
    }
    else
    {
        [self setParValuesOnBackNineScorecard:NO];
        [self setParValuesOnFrontNineScorecard:YES];
        [self setPlayerScoresOnBackNineScorecard:NO];
        [self setPlayerScoresOnFrontNineScorecard:YES];
    }
    _parGross.text = conV([_parSumInForNineHoles.text integerValue] + [_parSumOutForNineHoles.text integerValue]);
    _player1Gross.text = conV([_player1In.text integerValue] + [_player1Out.text integerValue]);
    _player2Gross.text = conV([_player2In.text integerValue] + [_player2Out.text integerValue]);
    _player3Gross.text = conV([_player3In.text integerValue] + [_player3Out.text integerValue]);
    _player4Gross.text = conV([_player4In.text integerValue] + [_player4Out.text integerValue]);
    [self calculateGrossScores];
}

- (void) viewDidLoad
{
    [super viewDidLoad];
    self.view.backgroundColor = [UIColor colorWithPatternImage:[UIImage imageNamed:@"background.png"]];
    [self initDAOs];
    [self initSwipers];
    [self setPlayerNamesOnFrontNineScorecard];
    [self buildScorecard];
}

NSString * (^ conV)(NSInteger val) = ^(NSInteger val) // Renamed due to linker error
{
    return [NSString stringWithFormat:@"%ld", (long) val];
};

- (void) initSwipers
{
    UISwipeGestureRecognizer * rightRecognizer = [[UISwipeGestureRecognizer alloc] initWithTarget:self action:@selector(rightSwipeHandler:)];
    
    [rightRecognizer setDirection:(UISwipeGestureRecognizerDirectionRight)];
    [self.view addGestureRecognizer:rightRecognizer];
    UISwipeGestureRecognizer * leftRecognizer = [[UISwipeGestureRecognizer alloc] initWithTarget:self action:@selector(leftSwipeHandler:)];
    [leftRecognizer setDirection:(UISwipeGestureRecognizerDirectionLeft)];
    [self.view addGestureRecognizer:leftRecognizer];
}

- (void) rightSwipeHandler:(UISwipeGestureRecognizer *)recognizer
{
    _isFrontNine = !_isFrontNine;
    [self performSegueWithIdentifier:@"showScorecard" sender:self];
    
}

- (void) leftSwipeHandler:(UISwipeGestureRecognizer *)recognizer
{
    _isFrontNine = !_isFrontNine;
    [self performSegueWithIdentifier:@"showScorecard" sender:self];
}

- (void) didReceiveMemoryWarning
{
    [super didReceiveMemoryWarning];
}

- (IBAction)unwindFromViewController:(UIStoryboardSegue *)segue
{
}

- (void) initDAOs
{
    if (_playerDAO == nil)
    {
        _playerDAO = [[PlayerDAO alloc] init];
        [_playerDAO initDb];
    }
    _players = [_playerDAO getPlayersOnScorecard];
    
    if (_teeColorDAO == nil)
    {
        _teeColorDAO = [[TeeColorDAO alloc] init];
        [_teeColorDAO initDb];
    }
    _teeColors = [_teeColorDAO getTeeColors];
    [_teeColors insertObject:[[TeeColor alloc] init] atIndex:0];
    
    if (_holeDAO == nil)
    {
        _holeDAO = [[HoleDAO alloc] init];
        [_holeDAO initDb];
    }
    
    if (_courseDAO == nil)
    {
        _courseDAO = [[CourseDAO alloc] init];
        [_courseDAO initDb];
    }
}


- (void) prepareForSegue:(UIStoryboardSegue *)segue sender:(id)sender
{
    if ([[segue identifier] isEqualToString:@"showScorecard"])
    {
        UINavigationController * navController = (UINavigationController *) [segue destinationViewController];
        ScoreViewController * vc = (ScoreViewController * ) [navController topViewController];
        [vc setIsFrontNine:_isFrontNine];
        [vc setGolfCourseName:_golfCourseName];
    }
}

- (void) setPlayerNamesOnFrontNineScorecard
{
    int i = 0;
    
    for (Player * p in _players)
    {
        ((UITextField *) _playerNames[i]).text = p.firstName;
        
        // Put name field background to reflect played tee color. Change font color to contrast.
        [((UITextField *) _playerNames[i])setBackgroundColor :[_teeColors[p.teeColor] getUIColorFromRGB]];
        const CGFloat * componentColors = CGColorGetComponents(([_teeColors[p.teeColor] getUIColorFromRGB]).CGColor);
        UIColor * contrastColor = [[UIColor alloc] initWithRed:(1.0 - componentColors[0])
                                                         green:(1.0 - componentColors[1])
                                                          blue:(1.0 - componentColors[2])
                                                         alpha:componentColors[3]];
        [((UITextField *) _playerNames[i++])setTextColor : contrastColor];
    }
}


- (void) setParValuesOnFrontNineScorecard:(BOOL)isParSumOnly
{
    NSMutableArray * holes = nil;
    int parSum = 0;
    BOOL done = NO;
    
    _parSumOutForNineHoles.text = @"0";
    
    for (TeeColor * t in _teeColors)
    {
        Course * course = [_courseDAO getSelectedCourse];
        holes = [_holeDAO getHoles:(int) course.courseId teeColorId:(int) t.teeColorId];
        
        int i = parSum = 0;
        
        for (Hole * h in holes)
        {
            if (h.par == 0)
            {
                break; // par values not typed in
            }
            if (!isParSumOnly)
            {
                ((UITextField *) _parValuesForNineHoles[i]).text = conV((int) h.par);
            }
            
            parSum += h.par;
            if (++i > 8)
            {
                _parSumOutForNineHoles.text = conV(parSum);
                done = YES;
                break;
            }
        }
        if (done)
        {
            break;
        }
    }
}

- (void) setParValuesOnBackNineScorecard:(BOOL)isParSumOnly
{
    NSMutableArray * holes = nil;
    int parSum = 0;
    BOOL done = NO;
    
    _parSumInForNineHoles.text = @"0";
    
    for (TeeColor * t in _teeColors)
    {
        if (t.teeColorId == 0)
        {
            continue;
        }
        
        Course * course = [_courseDAO getSelectedCourse];
        holes = [_holeDAO getHoles:(int) course.courseId teeColorId:(int) t.teeColorId];
        
        int i = parSum = 0;
        
        NSRange theRange;
        theRange.location = 9;
        theRange.length = 9;
        NSArray * lastNineHoles = [holes subarrayWithRange:theRange];
        
        for (Hole * h in lastNineHoles)
        {
            if (h.par == 0)
            {
                break; // par values not typed in
            }
            if (!isParSumOnly)
            {
                ((UITextField *) _parValuesForNineHoles[i]).text = conV((int) h.par);
            }
            parSum += h.par;
            if (++i > 8)
            {
                _parSumInForNineHoles.text = conV(parSum);
                done = YES;
                break;
            }
        }
        if (done)
        {
            break;
        }
    }
}


- (void) setHoleNumbers
{
    if (_isFrontNine)
    {
        for (int i = 0; i < 9; i++)
        {
            ((UITextField *) _holeNumbers[i]).text = conV(i + 1);
        }
    }
    else
    {
        for (int i = 0; i < 9; i++)
        {
            ((UITextField *) _holeNumbers[i]).text = conV(i + 10);
        }
        
    }
}

- (void) setPlayerScoresOnFrontNineScorecard:(BOOL)isScoreOnly
{
    for (Player * p in _players)
    {
        int total = 0;
        int validHoles = 9; // [MainViewController highestHolePlayed] > 8 ? 9 : [MainViewController highestHolePlayed];
        switch ([_players indexOfObject:p])
        {
            case 0:
                for (int i = 0; i < validHoles; i++)
                {
                    if (!isScoreOnly)
                    {
                        ((UITextField *) [_player1Strokes objectAtIndex:i]).text = [p getStrokesForHole:i + 1];
                    }
                    total += [[p getStrokesForHole:i + 1] integerValue];
                }
                _player1Out.text = conV(total);
                _player1Net.text = conV([_player1In.text integerValue] + [_player1Out.text integerValue] - p.hdcp);
                break;
                
            case 1:
                for (int i = 0; i < validHoles; i++)
                {
                    if (!isScoreOnly)
                    {
                        ((UITextField *) [_player2Strokes objectAtIndex:i]).text = [p getStrokesForHole:i + 1];
                    }
                    total += [[p getStrokesForHole:i + 1] integerValue];
                }
                _player2Out.text = conV(total);
                _player1Net.text = conV([_player2In.text integerValue] + [_player2Out.text integerValue] - p.hdcp);
                break;
                
            case 2:
                for (int i = 0; i < validHoles; i++)
                {
                    if (!isScoreOnly)
                    {
                        ((UITextField *) [_player3Strokes objectAtIndex:i]).text = [p getStrokesForHole:i + 1];
                    }
                    total += [[p getStrokesForHole:i + 1] integerValue];
                }
                _player3Out.text = conV(total);
                _player3Net.text = conV([_player3In.text integerValue] + [_player3Out.text integerValue] - p.hdcp);
                break;
                
            case 3:
                for (int i = 0; i < validHoles; i++)
                {
                    if (!isScoreOnly)
                    {
                        ((UITextField *) [_player4Strokes objectAtIndex:i]).text = [p getStrokesForHole:i + 1];
                    }
                    total += [[p getStrokesForHole:i + 1] integerValue];
                }
                _player4Out.text = conV(total);
                _player4Net.text = conV([_player4In.text integerValue] + [_player4Out.text integerValue] - p.hdcp);
                break;
            default:
                NSLog(@"Illegal player!");
                break;
        }
    }
}

- (void) setPlayerScoresOnBackNineScorecard:(BOOL)isScoreOnly
{
    for (Player * p in _players)
    {
        int total = 0;
        int validHoles = 9; // [MainViewController highestHolePlayed] < 9 ? 0 : [MainViewController highestHolePlayed] % 10;
        switch ([_players indexOfObject:p])
        {
            case 0:
                for (int i = 0; i < validHoles; i++)
                {
                    if (!isScoreOnly)
                    {
                        ((UITextField *) [_player1Strokes objectAtIndex:i]).text = [p getStrokesForHole:i + 10];
                    }
                    total += [[p getStrokesForHole:i + 10] integerValue];
                }
                _player1In.text = conV(total);
                _player1Net.text = conV([_player1In.text integerValue] + [_player1Out.text integerValue] - p.hdcp);
                break;
                
            case 1:
                for (int i = 0; i < validHoles; i++)
                {
                    if (!isScoreOnly)
                    {
                        ((UITextField *) [_player2Strokes objectAtIndex:i]).text = [p getStrokesForHole:i + 10];
                    }
                    total += [[p getStrokesForHole:i + 10] integerValue];
                }
                _player2In.text = conV(total);
                _player2Net.text = conV([_player2In.text integerValue] + [_player2Out.text integerValue] - p.hdcp);
                break;
                
            case 2:
                for (int i = 0; i < validHoles; i++)
                {
                    if (!isScoreOnly)
                    {
                        ((UITextField *) [_player3Strokes objectAtIndex:i]).text = [p getStrokesForHole:i + 10];
                    }
                    total += [[p getStrokesForHole:i + 10] integerValue];
                }
                _player3In.text = conV(total);
                _player3Net.text = conV([_player3In.text integerValue] + [_player3Out.text integerValue] - p.hdcp);
                break;
                
            case 3:
                for (int i = 0; i < validHoles; i++)
                {
                    if (!isScoreOnly)
                    {
                        ((UITextField *) [_player4Strokes objectAtIndex:i]).text = [p getStrokesForHole:i + 10];
                    }
                    total += [[p getStrokesForHole:i + 10] integerValue];
                }
                _player4In.text = conV(total);
                _player4Net.text = conV([_player4In.text integerValue] + [_player4Out.text integerValue] - p.hdcp);
                break;
            default:
                NSLog(@"Illegal player!");
                break;
        }
    }
}

- (void) showEmail:(NSString *)file
{
    NSString * emailTitle = _golfCourseName;
    NSString * messageBody = @"Here's a copy of the iScoreCard.";
    NSArray * toRecipents = [NSArray arrayWithObject:@"amyllykoski@yahoo.com"];
    
    MFMailComposeViewController * mc = [[MFMailComposeViewController alloc] init];
    
    mc.mailComposeDelegate = self;
    [mc setSubject:emailTitle];
    [mc setMessageBody:messageBody isHTML:YES];
    [mc setToRecipients:toRecipents];
    
    if (file != nil )
    {
        // Determine the file name and extension
        NSArray * filepart = [file componentsSeparatedByString:@"."];
        NSString * filename = [filepart objectAtIndex:0];
        NSString * extension = [filepart objectAtIndex:1];
        
        // Get the resource path and read the file using NSData
        NSString * filePath = [[NSBundle mainBundle] pathForResource:filename ofType:extension];
        NSData * fileData = [NSData dataWithContentsOfFile:filePath];
        
        // Determine the MIME type
        NSString * mimeType;
        if ([extension isEqualToString:@"jpg"])
        {
            mimeType = @"image/jpeg";
        }
        else if ([extension isEqualToString:@"png"])
        {
            mimeType = @"image/png";
        }
        else if ([extension isEqualToString:@"doc"])
        {
            mimeType = @"application/msword";
        }
        else if ([extension isEqualToString:@"ppt"])
        {
            mimeType = @"application/vnd.ms-powerpoint";
        }
        else if ([extension isEqualToString:@"html"])
        {
            mimeType = @"text/html";
        }
        else if ([extension isEqualToString:@"pdf"])
        {
            mimeType = @"application/pdf";
        }
        
        // Add attachment
        [mc addAttachmentData:fileData mimeType:mimeType fileName:filename];
    }
    
    // Present mail view controller on screen
    [self presentViewController:mc animated:YES completion:NULL];
    
}

- (void) mailComposeController:(MFMailComposeViewController *)controller didFinishWithResult:(MFMailComposeResult)result error:(NSError *)error
{
    switch (result)
    {
        case MFMailComposeResultCancelled:
            NSLog(@"Mail cancelled");
            break;
        case MFMailComposeResultSaved:
            NSLog(@"Mail saved");
            break;
        case MFMailComposeResultSent:
            NSLog(@"Mail sent");
            break;
        case MFMailComposeResultFailed:
            NSLog(@"Mail sent failure: %@", [error localizedDescription]);
            break;
        default:
            break;
    }
    
    // Close the Mail Interface
    [self dismissViewControllerAnimated:YES completion:NULL];
}

- (IBAction) sendScoreAsEmail:(id)sender
{
    UIScreen * screen = [UIScreen mainScreen];
    UIWindow * keyWindow = [[UIApplication sharedApplication] keyWindow];
    UIView * view = [screen snapshotViewAfterScreenUpdates:YES];
    
    UIGraphicsBeginImageContextWithOptions(screen.bounds.size, NO, 0);
    [keyWindow drawViewHierarchyInRect:keyWindow.bounds afterScreenUpdates:YES];
    UIImage * image = UIGraphicsGetImageFromCurrentImageContext();
    UIGraphicsEndImageContext();
    NSData * data = UIImagePNGRepresentation(image);
    [data writeToFile:[NSString stringWithFormat:@"%@/ScreenShot.png", [NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES) objectAtIndex:0]] atomically:YES];
    [self showEmail:@"ScreenShot.png"];
}
@end
