Ext.data.JsonP.profiles({"title":"Device Profiles","guide":"<h1>Using Device Profiles</h1>\n<div class='toc'>\n<p><strong>Contents</strong></p>\n<ol>\n<li><a href='#!/guide/profiles-section-1'>Setting up Profiles</a></li>\n<li><a href='#!/guide/profiles-section-2'>Determining the Active Profile</a></li>\n<li><a href='#!/guide/profiles-section-3'>The Launch Process</a></li>\n<li><a href='#!/guide/profiles-section-4'>Specializing Views</a></li>\n<li><a href='#!/guide/profiles-section-5'>Specializing Controllers</a></li>\n<li><a href='#!/guide/profiles-section-6'>Specializing Models</a></li>\n</ol>\n</div>\n\n<p>Today's mobile web applications are expected to work on a wide variety of devices, spanning from the smallest mobile phones to the largest tablets. These devices have a wide range of screen resolutions and are used in different ways. People tend to use phone apps while out of the house to rapidly gather some information or perform some action - often in under a minute. Tablet apps are more likely to be used for longer periods of time, usually within the home or somewhere else they can sit for a long time.</p>\n\n<p>All of this means that people expect different app experiences on different devices. However, much of your application logic and assets can be shared between these different experiences. Writing separate apps for each platform is time consuming, error-prone and just plain boring. Thankfully, Device Profiles give us a simple way to share just as much code as we need to between device types while making it easy to customize behavior, appearance and workflows for each device.</p>\n\n<h2 id='profiles-section-1'>Setting up Profiles</h2>\n\n<p>Device Profiles exist within the context of an <a href=\"#!/api/Ext.app.Application\" rel=\"Ext.app.Application\" class=\"docClass\">Application</a>, for example if we want to create an email app with phone and tablet profiles we can define our app.js like this (see the <a href=\"#!/guide/apps_intro\">Intro to Apps guide</a> if this is not familiar):</p>\n\n<pre><code><a href=\"#!/api/Ext-method-application\" rel=\"Ext-method-application\" class=\"docClass\">Ext.application</a>({\n    name: 'Mail',\n\n    profiles: ['Phone', 'Tablet']\n});\n</code></pre>\n\n<p>We didn't give our Application a <a href=\"#!/api/Ext.app.Application-cfg-launch\" rel=\"Ext.app.Application-cfg-launch\" class=\"docClass\">launch function</a>, so at the moment all it is going to do is load those two Profiles. By convention, it expects to find these in <em>app/profile/Phone.js</em> and <em>app/profile/Tablet.js</em>. Here's what our Phone profile might look like:</p>\n\n<pre><code><a href=\"#!/api/Ext-method-define\" rel=\"Ext-method-define\" class=\"docClass\">Ext.define</a>('Mail.profile.Phone', {\n    extend: '<a href=\"#!/api/Ext.app.Profile\" rel=\"Ext.app.Profile\" class=\"docClass\">Ext.app.Profile</a>',\n\n    config: {\n        name: 'Phone',\n        views: ['Main']\n    },\n\n    isActive: function() {\n        return Ext.os.is.Phone;\n    }\n});\n</code></pre>\n\n<p>The Tablet profile follows the same pattern. In our Phone profile we only really supplied three pieces of information - the Profile <a href=\"#!/api/Ext.app.Profile-cfg-name\" rel=\"Ext.app.Profile-cfg-name\" class=\"docClass\">name</a>, the optional set of additional <a href=\"#!/api/Ext.app.Profile-cfg-views\" rel=\"Ext.app.Profile-cfg-views\" class=\"docClass\">views</a> to load if this Profile is activated, and an <a href=\"#!/api/Ext.app.Profile-method-isActive\" rel=\"Ext.app.Profile-method-isActive\" class=\"docClass\">isActive</a> function.</p>\n\n<p>The <a href=\"#!/api/Ext.app.Profile-method-isActive\" rel=\"Ext.app.Profile-method-isActive\" class=\"docClass\">isActive</a> function is what determines if a given profile should be active on the device your app happens to be booting up on. By far the most common split is to create profiles for Phone and Tablet, using the built-in Ext.os.is.Phone and Ext.os.is.Tablet properties. You can write any code you like in the isActive function so long as it always returns true or false for the device it is running on.</p>\n\n<h2 id='profiles-section-2'>Determining the Active Profile</h2>\n\n<p>Once the Profiles have been loaded, their isActive functions are called in turn. The first one to return true is the Profile that the Application will boot with. This Profile is then set to the Application's <a href=\"#!/api/Ext.app.Application-cfg-currentProfile\" rel=\"Ext.app.Application-cfg-currentProfile\" class=\"docClass\">currentProfile</a>, and the Application prepares to load all of its dependencies - the models, views, controllers and other classes that constitute the app. It does this by combining its own dependencies with those specified in the active profile.</p>\n\n<p>For example, let's amend our Application so that it loads a few Models and Views of its own:</p>\n\n<pre><code><a href=\"#!/api/Ext-method-application\" rel=\"Ext-method-application\" class=\"docClass\">Ext.application</a>({\n    name: 'Mail',\n\n    profiles: ['Phone', 'Tablet'],\n\n    models: ['User'],\n    views: ['Navigation', 'Login']\n});\n</code></pre>\n\n<p>Now when we load the app on a phone, the Phone profile is activated and the application will load the following files:</p>\n\n<ul>\n<li>app/model/User.js</li>\n<li>app/view/Navigation.js</li>\n<li>app/view/Login.js</li>\n<li>app/view/<strong>phone</strong>/Main.js</li>\n</ul>\n\n\n<p>The first three items are specified in the Application itself - the User model plus the Navigation and Login views. The fourth item is specified by the Phone Profile and follows a special form. By convention, classes that are specific to a Profile are expected to be defined in a subdirectory with the same name as the Profile. For example, the 'Main' view specified in the Phone Profile will be loaded from <em>app/view/<strong>phone</strong>/Main.js</em>, whereas if we had defined 'Main' in the Application it would be loaded from <em>app/view/Main.js</em>.</p>\n\n<p>The same applies to all of the models, views, controllers and stores loaded in a Profile. This is important as it enables us to easily share behavior, view logic and more between profiles (see the specializing views and controllers sections below). If you need to load classes that don't fit with this convention, you can specify the full class name instead:</p>\n\n<pre><code><a href=\"#!/api/Ext-method-define\" rel=\"Ext-method-define\" class=\"docClass\">Ext.define</a>('Mail.profile.Phone', {\n    extend: '<a href=\"#!/api/Ext.app.Profile\" rel=\"Ext.app.Profile\" class=\"docClass\">Ext.app.Profile</a>',\n\n    config: {\n        name: 'Phone',\n        views: ['Main', 'Mail.view.SpecialView'],\n        models: ['Mail.model.Message']\n    },\n\n    isActive: function() {\n        return Ext.os.is.Phone;\n    }\n});\n</code></pre>\n\n<p>As we see above, you can mix and match fully-qualified class names (e.g. <em>'Mail.view.SomeView'</em>) and relatively specified class names (e.g. <em>'Main'</em>, which becomes <em>'Mail.view.phone.Main'</em>). Be aware that all models, views, controllers and stores specified for a Profile are treated this way. This means if there are Models or Stores that you want to load for Tablets only but do not want to make classes like <em>Mail.model.tablet.User</em>, you should specify the fully-qualified class names instead (e.g. <em>Mail.model.User</em> in this case).</p>\n\n<h2 id='profiles-section-3'>The Launch Process</h2>\n\n<p>The launch process using Profiles is almost exactly the same as it is without Profiles. Profile-based apps have a 3-stage launch process; after all of the dependencies have been loaded, the following happens:</p>\n\n<ol>\n<li>Controllers are instantiated; each Controller's <a href=\"#!/api/Ext.app.Controller-cfg-init\" rel=\"Ext.app.Controller-cfg-init\" class=\"docClass\">init</a> function is called</li>\n<li>The Profile's <a href=\"#!/api/Ext.app.Profile-method-launch\" rel=\"Ext.app.Profile-method-launch\" class=\"docClass\">launch</a> function is called</li>\n<li>The Application's <a href=\"#!/api/Ext.app.Application-cfg-launch\" rel=\"Ext.app.Application-cfg-launch\" class=\"docClass\">launch</a> function is called.</li>\n</ol>\n\n\n<p>When using Profiles it's common to use the Profile launch functions to create the app's initial UI. In many cases this means the Application's launch function is completely removed as the initial UI is usually different in each Profile (you can still specify an Application-wide launch function for setting up things like analytics or other profile-agnostic setup).</p>\n\n<p>A typical Profile launch function might look like this:</p>\n\n<pre><code><a href=\"#!/api/Ext-method-define\" rel=\"Ext-method-define\" class=\"docClass\">Ext.define</a>('Mail.profile.Phone', {\n    extend: '<a href=\"#!/api/Ext.app.Profile\" rel=\"Ext.app.Profile\" class=\"docClass\">Ext.app.Profile</a>',\n\n    config: {\n        name: 'Phone',\n        views: ['Main']\n    },\n\n    isActive: function() {\n        return Ext.os.is.Phone;\n    },\n\n    launch: function() {\n        <a href=\"#!/api/Ext-method-create\" rel=\"Ext-method-create\" class=\"docClass\">Ext.create</a>('Mail.view.phone.Main');\n    }\n});\n</code></pre>\n\n<p>Note that both Profile and Application launch functions are optional - if you don't define them they won't be called.</p>\n\n<h2 id='profiles-section-4'>Specializing Views</h2>\n\n<p>Most of the specialization in a Profile occurs in the views and the controllers. Let's look at the views first. Say we have a Tablet Profile that looks like this:</p>\n\n<pre><code><a href=\"#!/api/Ext-method-define\" rel=\"Ext-method-define\" class=\"docClass\">Ext.define</a>('Mail.profile.Tablet', {\n    extend: '<a href=\"#!/api/Ext.app.Profile\" rel=\"Ext.app.Profile\" class=\"docClass\">Ext.app.Profile</a>',\n\n    config: {\n        views: ['Main']\n    },\n\n    launch: function() {\n        <a href=\"#!/api/Ext-method-create\" rel=\"Ext-method-create\" class=\"docClass\">Ext.create</a>('Mail.view.tablet.Main');\n    }\n});\n</code></pre>\n\n<p>When we boot this app up on a tablet device, the file <em>app/views/tablet/Main.js</em> will be loaded as usual. Here's what we have in our <em>app/views/tablet/Main.js</em> file:</p>\n\n<pre><code><a href=\"#!/api/Ext-method-define\" rel=\"Ext-method-define\" class=\"docClass\">Ext.define</a>('Mail.view.tablet.Main', {\n    extend: 'Mail.view.Main',\n\n    config: {\n        title: 'Tablet-specific version'\n    }\n});\n</code></pre>\n\n<p>Usually when we define a view class we extend one of Sencha Touch's built in views but in this case we're extending Mail.view.Main - one of our own views. Here's how that on looks:</p>\n\n<pre><code><a href=\"#!/api/Ext-method-define\" rel=\"Ext-method-define\" class=\"docClass\">Ext.define</a>('Mail.view.Main', {\n    extend: '<a href=\"#!/api/Ext.Panel\" rel=\"Ext.Panel\" class=\"docClass\">Ext.Panel</a>',\n\n    config: {\n        title: 'Generic version',\n        html: 'This is the main screen'\n    }\n});\n</code></pre>\n\n<p>So we have a superclass (Mail.view.Main) and a Profile-specific subclass (Main.view.tablet.Main) which can customize any aspect of the superclass. In this case we're changing the title of the Main view from \"Generic version\" to \"Tablet-specific version\" in our subclass, so when our app launches that's what we will see.</p>\n\n<p>Because these are just normal classes it's easy to customize almost any part of the superclass using the flexible config system. For example, let's say we also have a phone version of the app - we could customize its version of the Main view like this (<em>app/view/phone/Main.js</em>):</p>\n\n<pre><code><a href=\"#!/api/Ext-method-define\" rel=\"Ext-method-define\" class=\"docClass\">Ext.define</a>('Mail.view.phone.Main', {\n    extend: 'Mail.view.Main',\n\n    config: {\n        title: 'Phone-specific version',\n\n        items: [\n            {\n                xtype: 'button',\n                text: 'This is a phone...'\n            }\n        ]\n    }\n});\n</code></pre>\n\n<h3>Sharing sub views</h3>\n\n<p>While the above is useful, it's more common to share certain pieces of views and stitch them together in different ways for different profiles. For example, imagine an email app where the tablet UI is a split screen with a message list on the left and the current message loaded on the right. The Phone version is the exact same message list and a similar message view, but this time in a <a href=\"#!/api/Ext.layout.Card\" rel=\"Ext.layout.Card\" class=\"docClass\">card layout</a> as there is not enough screen space to see them both simultaneously.</p>\n\n<p>To achieve this we'd start off creating the two shared sub views - the message list and the message viewer. In each case we've left the class config out for brevity:</p>\n\n<pre><code><a href=\"#!/api/Ext-method-define\" rel=\"Ext-method-define\" class=\"docClass\">Ext.define</a>('Main.view.MessageList', {\n    extend: '<a href=\"#!/api/Ext.dataview.List\" rel=\"Ext.dataview.List\" class=\"docClass\">Ext.List</a>',\n    xtype: 'messagelist',\n\n    //config goes here...\n});\n</code></pre>\n\n<p>And the Message Viewer:</p>\n\n<pre><code><a href=\"#!/api/Ext-method-define\" rel=\"Ext-method-define\" class=\"docClass\">Ext.define</a>('Main.view.MessageViewer', {\n    extend: '<a href=\"#!/api/Ext.Panel\" rel=\"Ext.Panel\" class=\"docClass\">Ext.Panel</a>',\n    xtype: 'messageviewer',\n\n    //config goes here...\n});\n</code></pre>\n\n<p>Now, to achieve our target layout the tablet Main view might do something like this:</p>\n\n<pre><code><a href=\"#!/api/Ext-method-define\" rel=\"Ext-method-define\" class=\"docClass\">Ext.define</a>('Main.view.tablet.Main', {\n    extend: '<a href=\"#!/api/Ext.Container\" rel=\"Ext.Container\" class=\"docClass\">Ext.Container</a>',\n\n    config: {\n        layout: 'fit',\n        items: [\n            {\n                xtype: 'messagelist',\n                width: 200,\n                docked: 'left'\n            },\n            {\n                xtype: 'messageviewer'\n            }\n        ]\n    }\n});\n</code></pre>\n\n<p>This will create a 200px wide messagelist on the left, and use the rest of the device's screen space to show the message viewer. Now let's say we want to achieve our Phone layout:</p>\n\n<pre><code><a href=\"#!/api/Ext-method-define\" rel=\"Ext-method-define\" class=\"docClass\">Ext.define</a>('Main.view.phone.Main', {\n    extend: '<a href=\"#!/api/Ext.Container\" rel=\"Ext.Container\" class=\"docClass\">Ext.Container</a>',\n\n    config: {\n        layout: 'card',\n        items: [\n            {\n                xtype: 'messagelist'\n            },\n            {\n                xtype: 'messageviewer'\n            }\n        ]\n    }\n});\n</code></pre>\n\n<p>In this case we're just using a Container with a <a href=\"#!/api/Ext.layout.Card\" rel=\"Ext.layout.Card\" class=\"docClass\">card layout</a> (a layout that only shows one item at a time), and putting both the list and the viewer into it. We'd need to plug in some logic that tells the Container to show the messageviewer when a message in the list is tapped on, but we've very easily reused our two sub views in different configurations based on which Profile is loaded.</p>\n\n<p>As before, we also have the option to customize the two shared views for each Profile - for example we could create <em>Mail.view.phone.MessageViewer</em> and <em>Mail.view.tablet.MessageViewer</em> subclasses, both of which extend the <em>Mail.view.MessageViewer</em> superclass. This enables us to again share a lot of view code between those classes while presenting customizations appropriate for the device the user happens to be using.</p>\n\n<h2 id='profiles-section-5'>Specializing Controllers</h2>\n\n<p>Just like with Views, many applications have a lot of Controller logic that can be shared across multiple Profiles. The biggest differences here between profiles are usually workflow-related. For example, an app's tablet profile may allow you to complete a workflow on a single page whereas the phone profile presents a multi-page wizard.</p>\n\n<p>Here we have a simple Phone profile that loads a view called <em>Main</em> and a controller called <em>Messages</em>. As before, this will load <em>app/view/phone/Main.js</em> and <em>app/controller/phone/Messages.js</em>:</p>\n\n<pre><code><a href=\"#!/api/Ext-method-define\" rel=\"Ext-method-define\" class=\"docClass\">Ext.define</a>('Mail.profile.Phone', {\n    extend: '<a href=\"#!/api/Ext.app.Profile\" rel=\"Ext.app.Profile\" class=\"docClass\">Ext.app.Profile</a>',\n\n    config: {\n        views: ['Main'],\n        controllers: ['Messages']\n    },\n\n    launch: function() {\n        <a href=\"#!/api/Ext-method-create\" rel=\"Ext-method-create\" class=\"docClass\">Ext.create</a>('Mail.view.phone.Main');\n    }\n});\n</code></pre>\n\n<p>Now, we know that our phone and tablet-specific controllers share most of their functionality so we'll start by creating a controller superclass in <em>app/controller/Messages.js</em>:</p>\n\n<pre><code><a href=\"#!/api/Ext-method-define\" rel=\"Ext-method-define\" class=\"docClass\">Ext.define</a>('Mail.controller.Messages', {\n    extend: '<a href=\"#!/api/Ext.app.Controller\" rel=\"Ext.app.Controller\" class=\"docClass\">Ext.app.Controller</a>',\n\n    config: {\n        refs: {\n            viewer: 'messageviewer',\n            messageList: 'messagelist'\n        },\n        control: {\n            messageList: {\n                itemtap: 'loadMessage'\n            }\n        }\n    },\n\n    loadMessage: function(item) {\n        this.getViewer().load(item);\n    }\n});\n</code></pre>\n\n<p>This Controller does 3 things:</p>\n\n<ol>\n<li>Sets up <a href=\"#!/api/Ext.app.Controller-cfg-refs\" rel=\"Ext.app.Controller-cfg-refs\" class=\"docClass\">refs</a> to views that we care about</li>\n<li>Listens for the itemtap event on the message list and calls the <em>loadMessage</em> function when itemtap is fired</li>\n<li>Loads the selected message item into the Viewer when <em>loadMessage</em> is called</li>\n</ol>\n\n\n<p>Now that we have this, it's easy to create our phone-specific controller:</p>\n\n<pre><code><a href=\"#!/api/Ext-method-define\" rel=\"Ext-method-define\" class=\"docClass\">Ext.define</a>('Mail.controller.phone.Messages', {\n    extend: 'Mail.controller.Messages',\n\n    config: {\n        refs: {\n            main: '#mainPanel'\n        }\n    },\n\n    loadMessage: function(item) {\n        this.callParent(arguments);\n        this.getMain().setActiveItem(1);\n    }\n});\n</code></pre>\n\n<p>Here we're extending the Messages superclass controller and providing 2 pieces of functionality:</p>\n\n<ol>\n<li>We add another ref for the phone UI's main panel</li>\n<li>We extend the loadMessage function to perform the original logic and then set the main panel's active item to the message viewer</li>\n</ol>\n\n\n<p>All of the configuration that was in the superclass is inherited by the subclass. In the case of duplicated configs like <a href=\"#!/api/Ext.app.Controller-cfg-refs\" rel=\"Ext.app.Controller-cfg-refs\" class=\"docClass\">refs</a>, the config is merged, so the phone Messages controller class has 3 refs - main, viewer and messageList. Just as with any other class that extends another, we can use callParent to extend an existing function in the superclass.</p>\n\n<p>Bear in mind that our <em>Mail.controller.Messages</em> superclass is not declared as a dependency by either the Application or the Profile. It it automatically loaded because our <em>Mail.controller.phone.Messages</em> controller extends it.</p>\n\n<h3>What to Share</h3>\n\n<p>In the example above we were able to share some (but not all) of our <a href=\"#!/api/Ext.app.Controller-cfg-refs\" rel=\"Ext.app.Controller-cfg-refs\" class=\"docClass\">refs</a>. We were also able to share the single event that we listen for with the Controller's <a href=\"#!/api/Ext.app.Controller-cfg-control\" rel=\"Ext.app.Controller-cfg-control\" class=\"docClass\">control</a> config. Generally speaking, the more the app diverges between profiles, the fewer refs and control configs you will be able to share.</p>\n\n<p>The one Controller config that should be shared across profiles is <a href=\"#!/api/Ext.app.Controller-cfg-routes\" rel=\"Ext.app.Controller-cfg-routes\" class=\"docClass\">routes</a>. These map urls to controller actions and allow for back button support and deep linking. It's important to keep the routes in the superclass because the same url should map to the same content regardless of the device.</p>\n\n<p>For example, if your friend is using the phone version of your app and sends you a link to the page she is currently on within your app, you should be able to tap that link on your tablet device and see the tablet-specific view for that url. Keeping all routes in the superclass enables you to keep a consistent url structure that will work regardless of device.</p>\n\n<h2 id='profiles-section-6'>Specializing Models</h2>\n\n<p>Models are customized per Profile less frequently than Controllers and Views, so don't usually require a subclass. In this case we just specify the fully qualified class names for models:</p>\n\n<pre><code><a href=\"#!/api/Ext-method-define\" rel=\"Ext-method-define\" class=\"docClass\">Ext.define</a>('Mail.profile.Tablet', {\n    extend: '<a href=\"#!/api/Ext.app.Profile\" rel=\"Ext.app.Profile\" class=\"docClass\">Ext.app.Profile</a>',\n\n    config: {\n        models: ['Mail.model.Group']\n    }\n});\n</code></pre>\n"});